<?php
/**
 * Installation Checker
 * Run this file to verify your setup is correct
 * Access via: http://localhost/payments/public/install-check.php
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<!DOCTYPE html>
<html>
<head>
    <title>Installation Check</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 800px; margin: 50px auto; padding: 20px; }
        .success { color: green; }
        .error { color: red; }
        .warning { color: orange; }
        .check-item { margin: 10px 0; padding: 10px; border-left: 3px solid #ccc; }
        .check-item.success { border-color: green; background: #f0fff0; }
        .check-item.error { border-color: red; background: #fff0f0; }
        .check-item.warning { border-color: orange; background: #fff8f0; }
        h1 { color: #333; }
        h2 { color: #666; margin-top: 30px; }
    </style>
</head>
<body>
    <h1>🔍 Payment System Installation Check</h1>";

$checks = [];
$allPassed = true;

// Check PHP Version
echo "<h2>PHP Requirements</h2>";
$phpVersion = phpversion();
if (version_compare($phpVersion, '7.4.0', '>=')) {
    echo "<div class='check-item success'>✅ PHP Version: {$phpVersion} (Required: 7.4+)</div>";
    $checks['php'] = true;
} else {
    echo "<div class='check-item error'>❌ PHP Version: {$phpVersion} (Required: 7.4+)</div>";
    $checks['php'] = false;
    $allPassed = false;
}

// Check PDO Extension
if (extension_loaded('pdo') && extension_loaded('pdo_mysql')) {
    echo "<div class='check-item success'>✅ PDO MySQL Extension: Installed</div>";
    $checks['pdo'] = true;
} else {
    echo "<div class='check-item error'>❌ PDO MySQL Extension: Not installed</div>";
    $checks['pdo'] = false;
    $allPassed = false;
}

// Check Database Connection
echo "<h2>Database Configuration</h2>";
$dbConfigPath = __DIR__ . '/../app/Config/db.php';
if (file_exists($dbConfigPath)) {
    echo "<div class='check-item success'>✅ Database config file exists</div>";
    
    try {
        $pdo = require $dbConfigPath;
        if ($pdo instanceof PDO) {
            echo "<div class='check-item success'>✅ Database connection: Successful</div>";
            $checks['db_connection'] = true;
            
            // Check if table exists
            $stmt = $pdo->query("SHOW TABLES LIKE 'transactions'");
            if ($stmt->rowCount() > 0) {
                echo "<div class='check-item success'>✅ Transactions table: Exists</div>";
                $checks['db_table'] = true;
            } else {
                echo "<div class='check-item error'>❌ Transactions table: Not found. Run migration!</div>";
                $checks['db_table'] = false;
                $allPassed = false;
            }
        } else {
            echo "<div class='check-item error'>❌ Database connection: Failed</div>";
            $checks['db_connection'] = false;
            $allPassed = false;
        }
    } catch (Exception $e) {
        echo "<div class='check-item error'>❌ Database connection: " . htmlspecialchars($e->getMessage()) . "</div>";
        $checks['db_connection'] = false;
        $allPassed = false;
    }
} else {
    echo "<div class='check-item error'>❌ Database config file: Not found at {$dbConfigPath}</div>";
    $checks['db_config'] = false;
    $allPassed = false;
}

// Check App Configuration
echo "<h2>Application Configuration</h2>";
$appConfigPath = __DIR__ . '/../app/Config/app.php';
if (file_exists($appConfigPath)) {
    echo "<div class='check-item success'>✅ App config file exists</div>";
    $appConfig = require $appConfigPath;
    
    if (isset($appConfig['bot_token']) && $appConfig['bot_token'] !== 'YOUR_TELEGRAM_BOT_TOKEN_HERE') {
        echo "<div class='check-item success'>✅ Telegram bot token: Configured</div>";
        $checks['telegram'] = true;
    } else {
        echo "<div class='check-item warning'>⚠️ Telegram bot token: Not configured (Optional for basic functionality)</div>";
        $checks['telegram'] = false;
    }
} else {
    echo "<div class='check-item error'>❌ App config file: Not found</div>";
    $checks['app_config'] = false;
    $allPassed = false;
}

// Check Directory Permissions
echo "<h2>File Permissions</h2>";
$storagePath = __DIR__ . '/../storage';
$logsPath = $storagePath . '/logs';
$cachePath = $storagePath . '/cache';

if (is_dir($storagePath)) {
    echo "<div class='check-item success'>✅ Storage directory exists</div>";
    
    if (is_writable($logsPath)) {
        echo "<div class='check-item success'>✅ Logs directory: Writable</div>";
        $checks['logs_writable'] = true;
    } else {
        echo "<div class='check-item warning'>⚠️ Logs directory: Not writable</div>";
        $checks['logs_writable'] = false;
    }
    
    if (is_writable($cachePath)) {
        echo "<div class='check-item success'>✅ Cache directory: Writable</div>";
        $checks['cache_writable'] = true;
    } else {
        echo "<div class='check-item warning'>⚠️ Cache directory: Not writable</div>";
        $checks['cache_writable'] = false;
    }
} else {
    echo "<div class='check-item error'>❌ Storage directory: Not found</div>";
    $checks['storage'] = false;
    $allPassed = false;
}

// Check Required Files
echo "<h2>Required Files</h2>";
$requiredFiles = [
    '../routes/web.php' => 'Routes file',
    '../app/Helpers/Parser.php' => 'Parser helper',
    '../app/Models/Transaction.php' => 'Transaction model',
    '../app/Controllers/WebhookController.php' => 'Webhook controller',
];

foreach ($requiredFiles as $file => $description) {
    $fullPath = __DIR__ . '/' . $file;
    if (file_exists($fullPath)) {
        echo "<div class='check-item success'>✅ {$description}: Exists</div>";
    } else {
        echo "<div class='check-item error'>❌ {$description}: Missing ({$file})</div>";
        $allPassed = false;
    }
}

// Final Summary
echo "<h2>Summary</h2>";
if ($allPassed && $checks['db_connection'] && $checks['db_table']) {
    echo "<div class='check-item success' style='font-size: 18px; font-weight: bold;'>
        ✅ All critical checks passed! Your system is ready to use.
    </div>";
    echo "<p><a href='index.php'>→ Go to Dashboard</a></p>";
} else {
    echo "<div class='check-item error' style='font-size: 18px; font-weight: bold;'>
        ❌ Some checks failed. Please review the errors above.
    </div>";
    echo "<p>📖 See <a href='../SETUP.md'>SETUP.md</a> for installation instructions.</p>";
}

echo "</body></html>";
?>

