<?php
/**
 * Database Migration Runner
 * Run this file to execute all migrations in order
 * Usage: php database/migrate.php
 */

require_once __DIR__ . '/../app/Config/db.php';

$pdo = require __DIR__ . '/../app/Config/db.php';

// Get database name from connection
$dbName = 'payment_system';

echo "🚀 Starting Database Migrations...\n\n";

$migrations = [
    'create_users_table.sql',
    'create_organizations_table.sql',
    'create_organization_users_table.sql',
    'create_password_resets_table.sql',
    'create_telegram_access_table.sql',
    'create_plans_table.sql',
    'create_subscriptions_table.sql',
    'update_transactions_table.sql',
    'add_superadmin_to_users_table.sql',
];

$seeders = [
    'seed_plans.sql',
    'seed_superadmin.sql',
];

$migrationDir = __DIR__ . '/migrations';
$seederDir = __DIR__ . '/seeders';

// Run migrations
echo "📦 Running Migrations:\n";
echo str_repeat("=", 50) . "\n";

foreach ($migrations as $migration) {
    $file = $migrationDir . '/' . $migration;
    
    if (!file_exists($file)) {
        echo "❌ Migration file not found: $migration\n";
        continue;
    }
    
    $sql = file_get_contents($file);
    
    // Skip if table already exists (for idempotency)
    $tableName = extractTableName($sql);
    if ($tableName && tableExists($pdo, $tableName)) {
        echo "⏭️  Skipping $migration (table already exists)\n";
        continue;
    }
    
    try {
        $pdo->exec($sql);
        echo "✅ Executed: $migration\n";
    } catch (PDOException $e) {
        // Check if it's just a "table already exists" error
        if (strpos($e->getMessage(), 'already exists') !== false || 
            strpos($e->getMessage(), 'Duplicate') !== false) {
            echo "⏭️  Skipped: $migration (already exists)\n";
        } else {
            echo "❌ Error in $migration: " . $e->getMessage() . "\n";
            echo "   SQL: " . substr($sql, 0, 100) . "...\n";
        }
    }
}

echo "\n";

// Run seeders
echo "🌱 Running Seeders:\n";
echo str_repeat("=", 50) . "\n";

foreach ($seeders as $seeder) {
    $file = $seederDir . '/' . $seeder;
    
    if (!file_exists($file)) {
        echo "❌ Seeder file not found: $seeder\n";
        continue;
    }
    
    // Special handling for superadmin seeder to use PHP password hashing
    if ($seeder === 'seed_superadmin.sql') {
        try {
            $passwordHash = password_hash('admin123', PASSWORD_DEFAULT);
            $sql = "INSERT INTO `users` (`name`, `email`, `password_hash`, `is_superadmin`, `email_verified_at`, `created_at`, `updated_at`) 
                    VALUES (
                      'Super Admin',
                      'admin@example.com',
                      :password_hash,
                      1,
                      NOW(),
                      NOW(),
                      NOW()
                    ) ON DUPLICATE KEY UPDATE 
                      `is_superadmin` = 1,
                      `updated_at` = NOW()";
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute([':password_hash' => $passwordHash]);
            echo "✅ Executed: $seeder (Super Admin: admin@example.com / admin123)\n";
        } catch (PDOException $e) {
            if (strpos($e->getMessage(), 'Duplicate') !== false) {
                echo "⏭️  Skipped: $seeder (superadmin already exists)\n";
            } else {
                echo "❌ Error in $seeder: " . $e->getMessage() . "\n";
            }
        }
        continue;
    }
    
    $sql = file_get_contents($file);
    
    try {
        $pdo->exec($sql);
        echo "✅ Executed: $seeder\n";
    } catch (PDOException $e) {
        // Check if it's just a duplicate entry error
        if (strpos($e->getMessage(), 'Duplicate') !== false) {
            echo "⏭️  Skipped: $seeder (data already exists)\n";
        } else {
            echo "❌ Error in $seeder: " . $e->getMessage() . "\n";
        }
    }
}

echo "\n";
echo "✨ Migration completed!\n";
echo "\n";
echo "📝 Default Super Admin Credentials:\n";
echo "   Email: admin@example.com\n";
echo "   Password: admin123\n";
echo "   ⚠️  Please change the password after first login!\n";
echo "\n";
echo "Next steps:\n";
echo "1. Login as superadmin at: http://localhost/payments/public/login.php\n";
echo "2. Register a new account at: http://localhost/payments/public/register.php\n";
echo "3. Configure your organization settings\n";

// Helper functions
function extractTableName($sql) {
    // Try to extract table name from CREATE TABLE statement
    if (preg_match('/CREATE TABLE\s+(?:IF NOT EXISTS\s+)?`?(\w+)`?/i', $sql, $matches)) {
        return $matches[1];
    }
    // Try ALTER TABLE
    if (preg_match('/ALTER TABLE\s+`?(\w+)`?/i', $sql, $matches)) {
        return $matches[1];
    }
    return null;
}

function tableExists($pdo, $tableName) {
    try {
        $stmt = $pdo->query("SHOW TABLES LIKE '$tableName'");
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        return false;
    }
}

