<?php

class Transaction
{
    private $pdo;
    private $table = "transactions";

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
    }

    // Insert new record
    public function create($data)
    {
        $sql = "INSERT INTO {$this->table}
            (organization_id, user_id, bank, payer_name, amount, date, time, method, merchant, reference_id, approval_code, raw_message)
            VALUES
            (:organization_id, :user_id, :bank, :payer_name, :amount, :date, :time, :method, :merchant, :reference_id, :approval_code, :raw_message)";

        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute($data);
    }

    // Fetch all with optional limit and offset (organization-scoped)
    public function all($limit = null, $offset = 0, $organizationId = null)
    {
        $sql = "SELECT * FROM {$this->table}";
        
        $params = [];
        if ($organizationId !== null) {
            $sql .= " WHERE organization_id = :organization_id";
            $params[':organization_id'] = $organizationId;
        }
        
        $sql .= " ORDER BY date DESC, time DESC";
        
        if ($limit !== null) {
            $sql .= " LIMIT :limit OFFSET :offset";
        }
        
        $stmt = $this->pdo->prepare($sql);
        if ($organizationId !== null) {
            $stmt->bindValue(':organization_id', (int)$organizationId, PDO::PARAM_INT);
        }
        if ($limit !== null) {
            $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
        }
        $stmt->execute();
        return $stmt->fetchAll();
    }

    // Get total count
    public function count($conditions = [])
    {
        $sql = "SELECT COUNT(*) as total FROM {$this->table}";
        $params = [];
        
        if (!empty($conditions)) {
            $where = [];
            
            if (isset($conditions['organization_id'])) {
                $where[] = "organization_id = :organization_id";
                $params[':organization_id'] = $conditions['organization_id'];
            }
            
            if (isset($conditions['date_from']) && isset($conditions['date_to'])) {
                $where[] = "date BETWEEN :date_from AND :date_to";
                $params[':date_from'] = $conditions['date_from'];
                $params[':date_to'] = $conditions['date_to'];
            }
            if (isset($conditions['bank'])) {
                $where[] = "bank = :bank";
                $params[':bank'] = $conditions['bank'];
            }
            if (isset($conditions['search'])) {
                $where[] = "(payer_name LIKE :search OR merchant LIKE :search OR reference_id LIKE :search)";
                $params[':search'] = '%' . $conditions['search'] . '%';
            }
            if (!empty($where)) {
                $sql .= " WHERE " . implode(" AND ", $where);
            }
        }
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetch()['total'];
    }

    // Filter by date range (organization-scoped)
    public function filter($start, $end, $organizationId = null)
    {
        $sql = "SELECT * FROM {$this->table} 
                WHERE date BETWEEN :start AND :end";
        
        $params = [
            ':start' => $start,
            ':end'   => $end
        ];

        if ($organizationId !== null) {
            $sql .= " AND organization_id = :organization_id";
            $params[':organization_id'] = $organizationId;
        }

        $sql .= " ORDER BY date DESC, time DESC";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll();
    }

    // Search transactions
    public function search($query, $limit = 100, $offset = 0)
    {
        $sql = "SELECT * FROM {$this->table} 
                WHERE payer_name LIKE :query 
                   OR merchant LIKE :query 
                   OR reference_id LIKE :query
                   OR bank LIKE :query
                ORDER BY date DESC, time DESC
                LIMIT :limit OFFSET :offset";

        $stmt = $this->pdo->prepare($sql);
        $stmt->bindValue(':query', '%' . $query . '%', PDO::PARAM_STR);
        $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    // Get statistics (organization-scoped)
    public function getStats($start = null, $end = null, $organizationId = null)
    {
        $where = [];
        $params = [];
        
        if ($organizationId !== null) {
            $where[] = "organization_id = :organization_id";
            $params[':organization_id'] = $organizationId;
        }
        
        if ($start && $end) {
            $where[] = "date BETWEEN :start AND :end";
            $params[':start'] = $start;
            $params[':end'] = $end;
        }
        
        $whereClause = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";

        $sql = "SELECT 
                    COUNT(*) as total_count,
                    SUM(amount) as total_amount,
                    AVG(amount) as avg_amount,
                    MIN(amount) as min_amount,
                    MAX(amount) as max_amount,
                    COUNT(DISTINCT bank) as bank_count,
                    COUNT(DISTINCT merchant) as merchant_count
                FROM {$this->table} {$whereClause}";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetch();
    }

    // Get statistics by bank (organization-scoped)
    public function getStatsByBank($start = null, $end = null, $organizationId = null)
    {
        $where = [];
        $params = [];
        
        if ($organizationId !== null) {
            $where[] = "organization_id = :organization_id";
            $params[':organization_id'] = $organizationId;
        }
        
        if ($start && $end) {
            $where[] = "date BETWEEN :start AND :end";
            $params[':start'] = $start;
            $params[':end'] = $end;
        }
        
        $whereClause = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";

        $sql = "SELECT 
                    bank,
                    COUNT(*) as count,
                    SUM(amount) as total,
                    AVG(amount) as avg_amount
                FROM {$this->table} {$whereClause}
                GROUP BY bank
                ORDER BY total DESC";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    // Get daily breakdown (organization-scoped)
    public function getDailyBreakdown($start, $end, $organizationId = null)
    {
        $sql = "SELECT 
                    date,
                    COUNT(*) as count,
                    SUM(amount) as total
                FROM {$this->table}
                WHERE date BETWEEN :start AND :end";
        
        $params = [
            ':start' => $start,
            ':end' => $end
        ];
        
        if ($organizationId !== null) {
            $sql .= " AND organization_id = :organization_id";
            $params[':organization_id'] = $organizationId;
        }
        
        $sql .= " GROUP BY date ORDER BY date ASC";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    // Get top merchants (organization-scoped)
    public function getTopMerchants($start = null, $end = null, $limit = 10, $organizationId = null)
    {
        $where = [];
        $params = [];
        
        if ($organizationId !== null) {
            $where[] = "organization_id = :organization_id";
            $params[':organization_id'] = $organizationId;
        }
        
        if ($start && $end) {
            $where[] = "date BETWEEN :start AND :end";
            $params[':start'] = $start;
            $params[':end'] = $end;
        }
        
        $whereClause = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";

        $sql = "SELECT 
                    merchant,
                    COUNT(*) as count,
                    SUM(amount) as total
                FROM {$this->table} {$whereClause}
                GROUP BY merchant
                ORDER BY total DESC
                LIMIT :limit";

        $stmt = $this->pdo->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    // Get recent transactions (organization-scoped)
    public function getRecent($limit = 10, $organizationId = null)
    {
        $sql = "SELECT * FROM {$this->table}";
        
        $params = [];
        if ($organizationId !== null) {
            $sql .= " WHERE organization_id = :organization_id";
            $params[':organization_id'] = $organizationId;
        }
        
        $sql .= " ORDER BY date DESC, time DESC LIMIT :limit";
        
        $stmt = $this->pdo->prepare($sql);
        if ($organizationId !== null) {
            $stmt->bindValue(':organization_id', (int)$organizationId, PDO::PARAM_INT);
        }
        $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }
}
