<?php

class PasswordReset
{
    private $pdo;
    private $table = "password_resets";

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
    }

    // Create reset token
    public function create($email, $token, $expiresInHours = 24)
    {
        // Delete existing tokens for this email
        $this->deleteByEmail($email);

        $expiresAt = date('Y-m-d H:i:s', strtotime("+{$expiresInHours} hours"));

        $sql = "INSERT INTO {$this->table} (email, token, expires_at) 
                VALUES (:email, :token, :expires_at)";
        
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([
            ':email' => $email,
            ':token' => $token,
            ':expires_at' => $expiresAt
        ]);
    }

    // Find valid token
    public function findValidToken($token)
    {
        $sql = "SELECT * FROM {$this->table} 
                WHERE token = :token 
                AND expires_at > NOW() 
                AND used_at IS NULL
                ORDER BY created_at DESC
                LIMIT 1";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':token' => $token]);
        return $stmt->fetch();
    }

    // Mark token as used
    public function markAsUsed($token)
    {
        $sql = "UPDATE {$this->table} 
                SET used_at = NOW() 
                WHERE token = :token";
        
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([':token' => $token]);
    }

    // Delete by email
    public function deleteByEmail($email)
    {
        $sql = "DELETE FROM {$this->table} WHERE email = :email";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([':email' => $email]);
    }

    // Clean expired tokens
    public function cleanExpired()
    {
        $sql = "DELETE FROM {$this->table} 
                WHERE expires_at < NOW() OR used_at IS NOT NULL";
        
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute();
    }

    // Generate random token
    public static function generateToken()
    {
        return bin2hex(random_bytes(32));
    }
}

