<?php

class Security
{
    /**
     * Generate CSRF token
     */
    public static function generateCSRFToken()
    {
        Session::start();
        if (!Session::has('csrf_token')) {
            Session::set('csrf_token', bin2hex(random_bytes(32)));
        }
        return Session::get('csrf_token');
    }

    /**
     * Verify CSRF token
     */
    public static function verifyCSRFToken($token)
    {
        Session::start();
        $sessionToken = Session::get('csrf_token');
        return $sessionToken && hash_equals($sessionToken, $token);
    }

    /**
     * Generate CSRF token field for forms
     */
    public static function csrfField()
    {
        $token = self::generateCSRFToken();
        return '<input type="hidden" name="csrf_token" value="' . htmlspecialchars($token, ENT_QUOTES, 'UTF-8') . '">';
    }

    /**
     * Verify CSRF token from POST request
     */
    public static function verifyCSRF()
    {
        $token = $_POST['csrf_token'] ?? '';
        if (!self::verifyCSRFToken($token)) {
            http_response_code(403);
            die('CSRF token validation failed. Please refresh the page and try again.');
        }
    }

    /**
     * Sanitize output to prevent XSS
     */
    public static function escape($value, $flags = ENT_QUOTES, $encoding = 'UTF-8')
    {
        if (is_null($value)) {
            return '';
        }
        return htmlspecialchars((string)$value, $flags, $encoding);
    }

    /**
     * Validate and sanitize email
     */
    public static function validateEmail($email)
    {
        return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
    }

    /**
     * Validate integer
     */
    public static function validateInt($value, $min = null, $max = null)
    {
        if (!is_numeric($value)) {
            return false;
        }
        $int = (int)$value;
        if ($min !== null && $int < $min) {
            return false;
        }
        if ($max !== null && $int > $max) {
            return false;
        }
        return true;
    }

    /**
     * Rate limiting - simple in-memory implementation
     */
    private static $rateLimitCache = [];

    public static function checkRateLimit($key, $maxAttempts = 5, $window = 300)
    {
        $now = time();
        $cacheKey = md5($key);
        
        if (!isset(self::$rateLimitCache[$cacheKey])) {
            self::$rateLimitCache[$cacheKey] = [
                'attempts' => 0,
                'window_start' => $now
            ];
        }

        $limit = &self::$rateLimitCache[$cacheKey];

        // Reset if window expired
        if ($now - $limit['window_start'] > $window) {
            $limit['attempts'] = 0;
            $limit['window_start'] = $now;
        }

        if ($limit['attempts'] >= $maxAttempts) {
            return false;
        }

        $limit['attempts']++;
        return true;
    }

    /**
     * Set security headers
     */
    public static function setSecurityHeaders()
    {
        // Prevent clickjacking
        header('X-Frame-Options: SAMEORIGIN');
        
        // Prevent MIME type sniffing
        header('X-Content-Type-Options: nosniff');
        
        // Enable XSS protection
        header('X-XSS-Protection: 1; mode=block');
        
        // Referrer policy
        header('Referrer-Policy: strict-origin-when-cross-origin');
        
        // Content Security Policy (basic)
        header("Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline'; img-src 'self' data: https:;");
    }

    /**
     * Regenerate session ID to prevent session fixation
     */
    public static function regenerateSession()
    {
        Session::start();
        if (Session::isAuthenticated()) {
            session_regenerate_id(true);
        }
    }
}
