<?php

class WebhookController
{
    private $pdo;
    private $transaction;

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
        $this->transaction = new Transaction($pdo);
    }

    public function handle()
    {
        try {
            $data = json_decode(file_get_contents("php://input"), true);

            if (!isset($data['message']['text'])) {
                Functions::logError("No message received.");
                return;
            }

            $message = $data['message']['text'];
            $chatId  = (string)$data['message']['chat']['id'];
            $chatType = isset($data['message']['chat']['type']) ? $data['message']['chat']['type'] : 'private';

            // Find organization by bot token (from webhook URL or message)
            // For now, we'll check all organizations with bot tokens
            $organizationModel = new Organization($this->pdo);
            $organizations = $organizationModel->all();
            
            $matchedOrg = null;
            $telegramAccess = new TelegramAccess($this->pdo);
            
            foreach ($organizations as $org) {
                if (empty($org['telegram_bot_token'])) {
                    continue;
                }
                
                // Check if chat has access to this organization
                if ($telegramAccess->hasAccess($org['id'], $chatId)) {
                    $matchedOrg = $org;
                    break;
                }
            }

            if (!$matchedOrg) {
                Functions::logError("No organization found for chat_id: $chatId");
                TelegramService::sendMessage($chatId, "❌ Access denied. Your chat is not authorized to send payments.");
                return;
            }

            $bank = Parser::detectBank($message);

            if ($bank === "ABA") {
                $parsed = Parser::parseABA($message);
            } elseif ($bank === "RHB") {
                $parsed = Parser::parseRHB($message);
            } else {
                TelegramService::sendMessage($chatId, "❌ Unknown payment format.", $matchedOrg['telegram_bot_token']);
                Functions::logError("Unknown format: " . $message);
                return;
            }

            if (!$parsed || empty($parsed['amount'])) {
                TelegramService::sendMessage($chatId, "⚠️ Failed to extract data.", $matchedOrg['telegram_bot_token']);
                Functions::logError("Parse failed: " . $message);
                return;
            }

            // Add raw message and organization/user info
            $parsed['raw_message'] = $message;
            $parsed['organization_id'] = $matchedOrg['id'];
            $parsed['user_id'] = null; // Could be linked to user if we track Telegram user IDs

            // Save to DB
            $this->transaction->create($parsed);

            // Confirm back to Telegram
            TelegramService::sendMessage($chatId, "✅ Payment recorded: {$parsed['amount']} USD", $matchedOrg['telegram_bot_token']);

        } catch (Exception $e) {
            Functions::logError("WEBHOOK ERROR: " . $e->getMessage());
        }
    }
}
