<?php

class AdminUsersController
{
    private $pdo;
    private $user;
    private $organization;

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
        $this->user = new User($pdo);
        $this->organization = new Organization($pdo);
    }

    // Show all users
    public function index()
    {
        $search = $_GET['search'] ?? '';
        $users = $this->getUsersWithStats($search);
        $organizationsWithWebhooks = $this->getOrganizationsWithWebhookInfo();
        $baseUrl = TelegramWebhookService::getBaseUrl();
        $webhookUrl = TelegramWebhookService::generateWebhookUrl($baseUrl);
        
        $title = "Manage Users";
        include __DIR__ . '/../../views/layouts/main.php';
        include __DIR__ . '/../../views/admin/users/index.php';
        include __DIR__ . '/../../views/layouts/footer.php';
    }

    // Show create form
    public function create()
    {
        $title = "Create User";
        include __DIR__ . '/../../views/layouts/main.php';
        include __DIR__ . '/../../views/admin/users/form.php';
        include __DIR__ . '/../../views/layouts/footer.php';
    }

    // Show edit form
    public function edit()
    {
        $id = $_GET['id'] ?? null;
        if (!$id) {
            Session::flash('error', 'User ID is required.');
            header('Location: admin-users.php');
            exit;
        }

        $user = $this->user->findById($id);
        if (!$user) {
            Session::flash('error', 'User not found.');
            header('Location: admin-users.php');
            exit;
        }

        // Get user organizations
        $organizations = $this->user->getOrganizations($id);

        $title = "Edit User";
        include __DIR__ . '/../../views/layouts/main.php';
        include __DIR__ . '/../../views/admin/users/form.php';
        include __DIR__ . '/../../views/layouts/footer.php';
    }

    // Store new user
    public function store()
    {
        $data = $this->validateAndPrepareData($_POST);
        
        if (isset($data['error'])) {
            Session::flash('error', $data['error']);
            header('Location: admin-users.php?action=create');
            exit;
        }

        // Check if email already exists
        if ($this->user->findByEmail($data['email'])) {
            Session::flash('error', 'Email already exists.');
            header('Location: admin-users.php?action=create');
            exit;
        }

        if ($userId = $this->user->create($data)) {
            Session::flash('success', 'User created successfully!');
            header('Location: admin-users.php');
        } else {
            Session::flash('error', 'Failed to create user. Please try again.');
            header('Location: admin-users.php?action=create');
        }
        exit;
    }

    // Update existing user
    public function update()
    {
        $id = $_POST['id'] ?? null;
        if (!$id) {
            Session::flash('error', 'User ID is required.');
            header('Location: admin-users.php');
            exit;
        }

        $user = $this->user->findById($id);
        if (!$user) {
            Session::flash('error', 'User not found.');
            header('Location: admin-users.php');
            exit;
        }

        $data = $this->validateAndPrepareData($_POST, true);
        
        if (isset($data['error'])) {
            Session::flash('error', $data['error']);
            header('Location: admin-users.php?action=edit&id=' . $id);
            exit;
        }

        // Check if email already exists (and not for this user)
        if (isset($data['email']) && $data['email'] !== $user['email']) {
            if ($this->user->findByEmail($data['email'])) {
                Session::flash('error', 'Email already exists.');
                header('Location: admin-users.php?action=edit&id=' . $id);
                exit;
            }
        }

        if ($this->user->update($id, $data)) {
            Session::flash('success', 'User updated successfully!');
            header('Location: admin-users.php');
        } else {
            Session::flash('error', 'Failed to update user. Please try again.');
            header('Location: admin-users.php?action=edit&id=' . $id);
        }
        exit;
    }

    // Delete user
    public function delete()
    {
        $id = $_POST['id'] ?? null;
        if (!$id) {
            Session::flash('error', 'User ID is required.');
            header('Location: admin-users.php');
            exit;
        }

        // Prevent deleting yourself
        $currentUser = Session::getUser();
        if ($currentUser && $currentUser['id'] == $id) {
            Session::flash('error', 'You cannot delete your own account.');
            header('Location: admin-users.php');
            exit;
        }

        if ($this->user->delete($id)) {
            Session::flash('success', 'User deleted successfully!');
        } else {
            Session::flash('error', 'Failed to delete user.');
        }

        header('Location: admin-users.php');
        exit;
    }

    // Set webhook for organization
    public function setWebhook()
    {
        $orgId = $_POST['organization_id'] ?? null;
        if (!$orgId) {
            Session::flash('error', 'Organization ID is required.');
            header('Location: admin-users.php');
            exit;
        }

        $organization = $this->organization->findById($orgId);
        if (!$organization || empty($organization['telegram_bot_token'])) {
            Session::flash('error', 'Organization not found or bot token not configured.');
            header('Location: admin-users.php');
            exit;
        }

        $baseUrl = $_POST['base_url'] ?? TelegramWebhookService::getBaseUrl();
        $webhookUrl = TelegramWebhookService::generateWebhookUrl($baseUrl);

        $result = TelegramWebhookService::setWebhook($organization['telegram_bot_token'], $webhookUrl);

        if ($result['success']) {
            Session::flash('success', 'Webhook set successfully! ' . ($result['message'] ?? ''));
        } else {
            Session::flash('error', 'Failed to set webhook: ' . ($result['error'] ?? 'Unknown error'));
        }

        header('Location: admin-users.php');
        exit;
    }

    // Delete webhook for organization
    public function deleteWebhook()
    {
        $orgId = $_POST['organization_id'] ?? null;
        if (!$orgId) {
            Session::flash('error', 'Organization ID is required.');
            header('Location: admin-users.php');
            exit;
        }

        $organization = $this->organization->findById($orgId);
        if (!$organization || empty($organization['telegram_bot_token'])) {
            Session::flash('error', 'Organization not found or bot token not configured.');
            header('Location: admin-users.php');
            exit;
        }

        $result = TelegramWebhookService::deleteWebhook($organization['telegram_bot_token']);

        if ($result['success']) {
            Session::flash('success', 'Webhook deleted successfully!');
        } else {
            Session::flash('error', 'Failed to delete webhook: ' . ($result['error'] ?? 'Unknown error'));
        }

        header('Location: admin-users.php');
        exit;
    }

    // Get users with statistics
    private function getUsersWithStats($search = '')
    {
        $sql = "SELECT u.*, 
                COUNT(DISTINCT ou.organization_id) as organization_count,
                COUNT(DISTINCT t.id) as transaction_count
                FROM users u
                LEFT JOIN organization_users ou ON u.id = ou.user_id
                LEFT JOIN organizations o ON ou.organization_id = o.id
                LEFT JOIN transactions t ON t.organization_id = o.id";

        $params = [];
        
        if (!empty($search)) {
            $sql .= " WHERE u.name LIKE :search OR u.email LIKE :search";
            $params[':search'] = "%{$search}%";
        }

        $sql .= " GROUP BY u.id ORDER BY u.created_at DESC";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    // Get organizations with webhook info
    public function getOrganizationsWithWebhookInfo()
    {
        $organizations = $this->organization->all();
        $orgsWithInfo = [];

        foreach ($organizations as $org) {
            $webhookInfo = null;
            if (!empty($org['telegram_bot_token'])) {
                $webhookInfo = TelegramWebhookService::getWebhookInfo($org['telegram_bot_token']);
            }
            
            $orgsWithInfo[] = [
                'organization' => $org,
                'webhook_info' => $webhookInfo
            ];
        }

        return $orgsWithInfo;
    }

    // Validate and prepare form data
    private function validateAndPrepareData($post, $isUpdate = false)
    {
        $name = trim($post['name'] ?? '');
        $email = trim($post['email'] ?? '');
        $password = $post['password'] ?? '';
        $passwordConfirm = $post['password_confirm'] ?? '';

        if (empty($name)) {
            return ['error' => 'Name is required.'];
        }

        if (empty($email)) {
            return ['error' => 'Email is required.'];
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return ['error' => 'Invalid email format.'];
        }

        $data = [
            'name' => $name,
            'email' => $email
        ];

        // Password is required for new users, optional for updates
        if (!$isUpdate || !empty($password)) {
            if (empty($password)) {
                return ['error' => 'Password is required.'];
            }

            if (strlen($password) < 8) {
                return ['error' => 'Password must be at least 8 characters.'];
            }

            if ($password !== $passwordConfirm) {
                return ['error' => 'Passwords do not match.'];
            }

            $data['password'] = $password;
        }

        return $data;
    }
}

