-- ============================================
-- Complete Database Setup: Tables + Seed Data
-- ============================================
-- This file contains ALL table creation statements AND seed data
-- Run this file to set up the entire database from scratch
-- ============================================

-- ============================================
-- SECTION 1: Create Tables
-- ============================================

-- 1.1 Create users table
CREATE TABLE IF NOT EXISTS `users` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `name` varchar(255) NOT NULL,
  `email` varchar(255) NOT NULL UNIQUE,
  `password_hash` varchar(255) NOT NULL,
  `email_verified_at` datetime DEFAULT NULL,
  `remember_token` varchar(100) DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_email` (`email`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 1.2 Create organizations table
CREATE TABLE IF NOT EXISTS `organizations` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `name` varchar(255) NOT NULL,
  `slug` varchar(255) NOT NULL UNIQUE,
  `telegram_bot_token` varchar(255) DEFAULT NULL,
  `settings` text DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_slug` (`slug`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 1.3 Create organization_users table
CREATE TABLE IF NOT EXISTS `organization_users` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `organization_id` int(11) NOT NULL,
  `user_id` int(11) NOT NULL,
  `role` enum('owner','admin','member','viewer') NOT NULL DEFAULT 'member',
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_org_user` (`organization_id`, `user_id`),
  KEY `idx_organization_id` (`organization_id`),
  KEY `idx_user_id` (`user_id`),
  FOREIGN KEY (`organization_id`) REFERENCES `organizations`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 1.4 Create password_resets table
CREATE TABLE IF NOT EXISTS `password_resets` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `email` varchar(255) NOT NULL,
  `token` varchar(255) NOT NULL,
  `expires_at` datetime NOT NULL,
  `used_at` datetime DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_email` (`email`),
  KEY `idx_token` (`token`),
  KEY `idx_expires_at` (`expires_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 1.5 Create telegram_access table
CREATE TABLE IF NOT EXISTS `telegram_access` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `organization_id` int(11) NOT NULL,
  `chat_id` varchar(255) NOT NULL,
  `chat_type` enum('user','group','channel') NOT NULL DEFAULT 'user',
  `chat_title` varchar(255) DEFAULT NULL,
  `is_active` tinyint(1) NOT NULL DEFAULT 1,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_org_chat` (`organization_id`, `chat_id`),
  KEY `idx_organization_id` (`organization_id`),
  KEY `idx_chat_id` (`chat_id`),
  FOREIGN KEY (`organization_id`) REFERENCES `organizations`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 1.6 Create plans table
CREATE TABLE IF NOT EXISTS `plans` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `name` varchar(100) NOT NULL,
  `slug` varchar(100) NOT NULL UNIQUE,
  `description` text,
  `price_monthly` decimal(10,2) NOT NULL,
  `price_annual` decimal(10,2) NOT NULL,
  `features` text DEFAULT NULL,
  `max_transactions` int(11) DEFAULT NULL,
  `max_users` int(11) DEFAULT NULL,
  `max_organizations` int(11) DEFAULT 1,
  `is_active` tinyint(1) NOT NULL DEFAULT 1,
  `sort_order` int(11) DEFAULT 0,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_slug` (`slug`),
  KEY `idx_is_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 1.7 Create subscriptions table
CREATE TABLE IF NOT EXISTS `subscriptions` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `organization_id` int(11) NOT NULL,
  `plan_id` int(11) NOT NULL,
  `billing_period` enum('monthly','annual') NOT NULL DEFAULT 'monthly',
  `status` enum('active','cancelled','expired','trial') NOT NULL DEFAULT 'trial',
  `starts_at` datetime NOT NULL,
  `ends_at` datetime NOT NULL,
  `cancelled_at` datetime DEFAULT NULL,
  `trial_ends_at` datetime DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_organization_id` (`organization_id`),
  KEY `idx_plan_id` (`plan_id`),
  KEY `idx_status` (`status`),
  FOREIGN KEY (`organization_id`) REFERENCES `organizations`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`plan_id`) REFERENCES `plans`(`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- 1.8 Create transactions table
CREATE TABLE IF NOT EXISTS `transactions` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `bank` varchar(20) NOT NULL,
  `payer_name` varchar(255) NOT NULL,
  `amount` decimal(10,2) NOT NULL,
  `date` date NOT NULL,
  `time` time NOT NULL,
  `method` varchar(100),
  `merchant` varchar(255),
  `reference_id` varchar(255),
  `approval_code` varchar(255),
  `raw_message` text,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================
-- SECTION 2: Update Tables (Add Columns/Indexes)
-- ============================================

-- 2.1 Add organization_id and user_id to transactions table
SET @col_exists = 0;
SELECT COUNT(*) INTO @col_exists 
FROM INFORMATION_SCHEMA.COLUMNS 
WHERE TABLE_SCHEMA = DATABASE() 
  AND TABLE_NAME = 'transactions' 
  AND COLUMN_NAME = 'organization_id';

SET @sql = IF(@col_exists = 0,
  'ALTER TABLE `transactions` ADD COLUMN `organization_id` int(11) DEFAULT NULL AFTER `id`',
  'SELECT "Column organization_id already exists" AS message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @col_exists = 0;
SELECT COUNT(*) INTO @col_exists 
FROM INFORMATION_SCHEMA.COLUMNS 
WHERE TABLE_SCHEMA = DATABASE() 
  AND TABLE_NAME = 'transactions' 
  AND COLUMN_NAME = 'user_id';

SET @sql = IF(@col_exists = 0,
  'ALTER TABLE `transactions` ADD COLUMN `user_id` int(11) DEFAULT NULL AFTER `organization_id`',
  'SELECT "Column user_id already exists" AS message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add indexes to transactions table
SET @idx_exists = 0;
SELECT COUNT(*) INTO @idx_exists 
FROM INFORMATION_SCHEMA.STATISTICS 
WHERE TABLE_SCHEMA = DATABASE() 
  AND TABLE_NAME = 'transactions' 
  AND INDEX_NAME = 'idx_organization_id';

SET @sql = IF(@idx_exists = 0,
  'ALTER TABLE `transactions` ADD KEY `idx_organization_id` (`organization_id`)',
  'SELECT "Index idx_organization_id already exists" AS message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @idx_exists = 0;
SELECT COUNT(*) INTO @idx_exists 
FROM INFORMATION_SCHEMA.STATISTICS 
WHERE TABLE_SCHEMA = DATABASE() 
  AND TABLE_NAME = 'transactions' 
  AND INDEX_NAME = 'idx_user_id';

SET @sql = IF(@idx_exists = 0,
  'ALTER TABLE `transactions` ADD KEY `idx_user_id` (`user_id`)',
  'SELECT "Index idx_user_id already exists" AS message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add foreign keys to transactions table
SET @fk_exists = 0;
SELECT COUNT(*) INTO @fk_exists 
FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
WHERE TABLE_SCHEMA = DATABASE() 
  AND TABLE_NAME = 'transactions' 
  AND CONSTRAINT_NAME = 'transactions_ibfk_1';

SET @sql = IF(@fk_exists = 0,
  'ALTER TABLE `transactions` ADD FOREIGN KEY (`organization_id`) REFERENCES `organizations`(`id`) ON DELETE CASCADE',
  'SELECT "Foreign key for organization_id already exists" AS message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @fk_exists = 0;
SELECT COUNT(*) INTO @fk_exists 
FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
WHERE TABLE_SCHEMA = DATABASE() 
  AND TABLE_NAME = 'transactions' 
  AND CONSTRAINT_NAME = 'transactions_ibfk_2';

SET @sql = IF(@fk_exists = 0,
  'ALTER TABLE `transactions` ADD FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE SET NULL',
  'SELECT "Foreign key for user_id already exists" AS message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- 2.2 Add is_superadmin field to users table
SET @col_exists = 0;
SELECT COUNT(*) INTO @col_exists 
FROM INFORMATION_SCHEMA.COLUMNS 
WHERE TABLE_SCHEMA = DATABASE() 
  AND TABLE_NAME = 'users' 
  AND COLUMN_NAME = 'is_superadmin';

SET @sql = IF(@col_exists = 0,
  'ALTER TABLE `users` ADD COLUMN `is_superadmin` tinyint(1) NOT NULL DEFAULT 0 AFTER `email_verified_at`',
  'SELECT "Column is_superadmin already exists" AS message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add index for superadmin queries
SET @idx_exists = 0;
SELECT COUNT(*) INTO @idx_exists 
FROM INFORMATION_SCHEMA.STATISTICS 
WHERE TABLE_SCHEMA = DATABASE() 
  AND TABLE_NAME = 'users' 
  AND INDEX_NAME = 'idx_is_superadmin';

SET @sql = IF(@idx_exists = 0,
  'ALTER TABLE `users` ADD KEY `idx_is_superadmin` (`is_superadmin`)',
  'SELECT "Index idx_is_superadmin already exists" AS message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- ============================================
-- SECTION 3: Insert Seed Data
-- ============================================

-- 3.1 Insert default plans
INSERT IGNORE INTO `plans` (`name`, `slug`, `description`, `price_monthly`, `price_annual`, `features`, `max_transactions`, `max_users`, `max_organizations`, `sort_order`) VALUES
('Starter', 'starter', 'Perfect for small businesses getting started', 9.99, 99.99, '["Up to 1,000 transactions/month", "1 organization", "Up to 3 team members", "Basic reports", "Email support"]', 1000, 3, 1, 1),
('Professional', 'professional', 'For growing businesses with advanced needs', 29.99, 299.99, '["Up to 10,000 transactions/month", "3 organizations", "Up to 10 team members", "Advanced reports & analytics", "Priority support", "API access", "Custom integrations"]', 10000, 10, 3, 2),
('Enterprise', 'enterprise', 'For large organizations with unlimited needs', 99.99, 999.99, '["Unlimited transactions", "Unlimited organizations", "Unlimited team members", "Custom reports & dashboards", "24/7 priority support", "Full API access", "Custom integrations", "Dedicated account manager", "SLA guarantee"]', NULL, NULL, NULL, 3);

-- 3.2 Insert superadmin user
-- NOTE: For manual SQL imports, generate a password hash using: password_hash('admin123', PASSWORD_DEFAULT)
-- Default password: admin123 (change after first login)
-- Email: admin@example.com (change as needed)
--
-- For manual import, replace the password_hash below with a generated hash:
-- Example: php -r "echo password_hash('admin123', PASSWORD_DEFAULT);"

INSERT INTO `users` (`name`, `email`, `password_hash`, `is_superadmin`, `email_verified_at`, `created_at`, `updated_at`) 
VALUES (
  'Super Admin',
  'admin@example.com',
  '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', -- Replace with generated hash for manual import
  1,
  NOW(),
  NOW(),
  NOW()
) ON DUPLICATE KEY UPDATE 
  `is_superadmin` = 1,
  `updated_at` = NOW();

-- ============================================
-- End of Complete Database Setup
-- ============================================
-- 
-- Default Super Admin Credentials:
-- Email: admin@example.com
-- Password: admin123
-- ⚠️  Please change the password after first login!
-- ============================================
