<?php

class TelegramWebhookService
{
    /**
     * Set webhook for a bot token
     */
    public static function setWebhook($botToken, $webhookUrl)
    {
        if (empty($botToken) || empty($webhookUrl)) {
            return ['success' => false, 'error' => 'Bot token and webhook URL are required.'];
        }

        $url = "https://api.telegram.org/bot{$botToken}/setWebhook";
        $params = http_build_query([
            'url' => $webhookUrl
        ]);

        $context = stream_context_create([
            'http' => [
                'method' => 'POST',
                'header' => 'Content-Type: application/x-www-form-urlencoded',
                'content' => $params,
                'timeout' => 10
            ]
        ]);

        $result = @file_get_contents($url, false, $context);
        
        if ($result === false) {
            return ['success' => false, 'error' => 'Failed to connect to Telegram API.'];
        }

        $response = json_decode($result, true);
        
        if (isset($response['ok']) && $response['ok'] === true) {
            return [
                'success' => true,
                'message' => $response['description'] ?? 'Webhook set successfully.',
                'result' => $response['result'] ?? null
            ];
        }

        return [
            'success' => false,
            'error' => $response['description'] ?? 'Failed to set webhook.'
        ];
    }

    /**
     * Delete webhook for a bot token
     */
    public static function deleteWebhook($botToken)
    {
        if (empty($botToken)) {
            return ['success' => false, 'error' => 'Bot token is required.'];
        }

        $url = "https://api.telegram.org/bot{$botToken}/deleteWebhook";
        $params = http_build_query(['drop_pending_updates' => true]);

        $context = stream_context_create([
            'http' => [
                'method' => 'POST',
                'header' => 'Content-Type: application/x-www-form-urlencoded',
                'content' => $params,
                'timeout' => 10
            ]
        ]);

        $result = @file_get_contents($url, false, $context);
        
        if ($result === false) {
            return ['success' => false, 'error' => 'Failed to connect to Telegram API.'];
        }

        $response = json_decode($result, true);
        
        if (isset($response['ok']) && $response['ok'] === true) {
            return [
                'success' => true,
                'message' => $response['description'] ?? 'Webhook deleted successfully.'
            ];
        }

        return [
            'success' => false,
            'error' => $response['description'] ?? 'Failed to delete webhook.'
        ];
    }

    /**
     * Get webhook info for a bot token
     */
    public static function getWebhookInfo($botToken)
    {
        if (empty($botToken)) {
            return ['success' => false, 'error' => 'Bot token is required.'];
        }

        $url = "https://api.telegram.org/bot{$botToken}/getWebhookInfo";

        $context = stream_context_create([
            'http' => [
                'method' => 'GET',
                'timeout' => 10
            ]
        ]);

        $result = @file_get_contents($url, false, $context);
        
        if ($result === false) {
            return ['success' => false, 'error' => 'Failed to connect to Telegram API.'];
        }

        $response = json_decode($result, true);
        
        if (isset($response['ok']) && $response['ok'] === true) {
            return [
                'success' => true,
                'webhook' => $response['result'] ?? null
            ];
        }

        return [
            'success' => false,
            'error' => $response['description'] ?? 'Failed to get webhook info.'
        ];
    }

    /**
     * Get base URL for webhook
     */
    public static function getBaseUrl()
    {
        // Try to get from environment/config first
        $config = require __DIR__ . '/../Config/app.php';
        $baseUrl = $config['app_url'] ?? '';
        
        if (!empty($baseUrl)) {
            return rtrim($baseUrl, '/');
        }

        // Auto-detect from current request
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $scriptPath = dirname($_SERVER['SCRIPT_NAME'] ?? '');
        
        // Remove /public if present in path
        $scriptPath = str_replace('/public', '', $scriptPath);
        
        return $protocol . '://' . $host . $scriptPath;
    }

    /**
     * Generate webhook URL
     */
    public static function generateWebhookUrl($baseUrl = null)
    {
        if ($baseUrl === null) {
            $baseUrl = self::getBaseUrl();
        }
        
        return rtrim($baseUrl, '/') . '/webhook.php';
    }
}

