<?php

class TelegramAccess
{
    private $pdo;
    private $table = "telegram_access";

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
    }

    // Create access record
    public function create($data)
    {
        $sql = "INSERT INTO {$this->table} (organization_id, chat_id, chat_type, chat_title, is_active) 
                VALUES (:organization_id, :chat_id, :chat_type, :chat_title, :is_active)";
        
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([
            ':organization_id' => $data['organization_id'],
            ':chat_id' => $data['chat_id'],
            ':chat_type' => $data['chat_type'] ?? 'user',
            ':chat_title' => $data['chat_title'] ?? null,
            ':is_active' => $data['is_active'] ?? 1
        ]);
    }

    // Find by organization and chat_id
    public function findByChatId($organizationId, $chatId)
    {
        $sql = "SELECT * FROM {$this->table} 
                WHERE organization_id = :organization_id AND chat_id = :chat_id AND is_active = 1
                LIMIT 1";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([
            ':organization_id' => $organizationId,
            ':chat_id' => $chatId
        ]);
        return $stmt->fetch();
    }

    // Check if chat has access
    public function hasAccess($organizationId, $chatId)
    {
        $access = $this->findByChatId($organizationId, $chatId);
        return $access !== false;
    }

    // Get all access for organization
    public function getByOrganization($organizationId)
    {
        $sql = "SELECT * FROM {$this->table} 
                WHERE organization_id = :organization_id
                ORDER BY created_at DESC";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':organization_id' => $organizationId]);
        return $stmt->fetchAll();
    }

    // Update access
    public function update($id, $data)
    {
        $fields = [];
        $params = [':id' => $id];

        if (isset($data['chat_id'])) {
            $fields[] = "chat_id = :chat_id";
            $params[':chat_id'] = $data['chat_id'];
        }
        if (isset($data['chat_type'])) {
            $fields[] = "chat_type = :chat_type";
            $params[':chat_type'] = $data['chat_type'];
        }
        if (isset($data['chat_title'])) {
            $fields[] = "chat_title = :chat_title";
            $params[':chat_title'] = $data['chat_title'];
        }
        if (isset($data['is_active'])) {
            $fields[] = "is_active = :is_active";
            $params[':is_active'] = $data['is_active'] ? 1 : 0;
        }

        if (empty($fields)) {
            return false;
        }

        $sql = "UPDATE {$this->table} SET " . implode(", ", $fields) . " WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute($params);
    }

    // Delete access
    public function delete($id)
    {
        $sql = "DELETE FROM {$this->table} WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([':id' => $id]);
    }

    // Toggle active status
    public function toggleActive($id)
    {
        $sql = "UPDATE {$this->table} SET is_active = NOT is_active WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([':id' => $id]);
    }

}

