<?php

class Subscription
{
    private $pdo;
    private $table = "subscriptions";

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
    }

    // Create subscription
    public function create($data)
    {
        $sql = "INSERT INTO {$this->table} 
                (organization_id, plan_id, billing_period, status, starts_at, ends_at, trial_ends_at) 
                VALUES 
                (:organization_id, :plan_id, :billing_period, :status, :starts_at, :ends_at, :trial_ends_at)";
        
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([
            ':organization_id' => $data['organization_id'],
            ':plan_id' => $data['plan_id'],
            ':billing_period' => $data['billing_period'] ?? 'monthly',
            ':status' => $data['status'] ?? 'trial',
            ':starts_at' => $data['starts_at'],
            ':ends_at' => $data['ends_at'],
            ':trial_ends_at' => $data['trial_ends_at'] ?? null
        ]);
    }

    // Get active subscription for organization
    public function getActive($organizationId)
    {
        $sql = "SELECT s.*, p.name as plan_name, p.slug as plan_slug, p.price_monthly, p.price_annual
                FROM {$this->table} s
                INNER JOIN plans p ON s.plan_id = p.id
                WHERE s.organization_id = :organization_id 
                AND s.status IN ('active', 'trial')
                AND s.ends_at > NOW()
                ORDER BY s.created_at DESC
                LIMIT 1";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':organization_id' => $organizationId]);
        return $stmt->fetch();
    }

    // Get all subscriptions for organization
    public function getByOrganization($organizationId)
    {
        $sql = "SELECT s.*, p.name as plan_name, p.slug as plan_slug
                FROM {$this->table} s
                INNER JOIN plans p ON s.plan_id = p.id
                WHERE s.organization_id = :organization_id
                ORDER BY s.created_at DESC";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':organization_id' => $organizationId]);
        return $stmt->fetchAll();
    }

    // Update subscription
    public function update($id, $data)
    {
        $fields = [];
        $params = [':id' => $id];

        $allowed = ['plan_id', 'billing_period', 'status', 'starts_at', 'ends_at', 'cancelled_at', 'trial_ends_at'];
        
        foreach ($allowed as $field) {
            if (isset($data[$field])) {
                $fields[] = "$field = :$field";
                $params[":$field"] = $data[$field];
            }
        }

        if (empty($fields)) {
            return false;
        }

        $sql = "UPDATE {$this->table} SET " . implode(", ", $fields) . " WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute($params);
    }

    // Cancel subscription
    public function cancel($id)
    {
        $sql = "UPDATE {$this->table} 
                SET status = 'cancelled', cancelled_at = NOW() 
                WHERE id = :id";
        
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([':id' => $id]);
    }

    // Check if organization has active subscription
    public function hasActive($organizationId)
    {
        $subscription = $this->getActive($organizationId);
        return $subscription !== false;
    }

    // Get trial subscription
    public function createTrial($organizationId, $planId, $days = 14)
    {
        $startsAt = date('Y-m-d H:i:s');
        $trialEndsAt = date('Y-m-d H:i:s', strtotime("+{$days} days"));
        
        return $this->create([
            'organization_id' => $organizationId,
            'plan_id' => $planId,
            'billing_period' => 'monthly',
            'status' => 'trial',
            'starts_at' => $startsAt,
            'ends_at' => $trialEndsAt,
            'trial_ends_at' => $trialEndsAt
        ]);
    }

    // Activate subscription (convert from trial or new)
    public function activate($id, $billingPeriod = 'monthly')
    {
        $subscription = $this->findById($id);
        if (!$subscription) {
            return false;
        }

        $plan = new Plan($this->pdo);
        $planData = $plan->findById($subscription['plan_id']);
        
        $startsAt = date('Y-m-d H:i:s');
        $endsAt = $billingPeriod === 'annual' 
            ? date('Y-m-d H:i:s', strtotime('+1 year'))
            : date('Y-m-d H:i:s', strtotime('+1 month'));

        return $this->update($id, [
            'status' => 'active',
            'billing_period' => $billingPeriod,
            'starts_at' => $startsAt,
            'ends_at' => $endsAt
        ]);
    }

    // Find by ID
    public function findById($id)
    {
        $sql = "SELECT * FROM {$this->table} WHERE id = :id LIMIT 1";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':id' => $id]);
        return $stmt->fetch();
    }
}

