<?php

class Plan
{
    private $pdo;
    private $table = "plans";

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
    }

    // Get all active plans
    public function all($activeOnly = true)
    {
        $sql = "SELECT * FROM {$this->table}";
        if ($activeOnly) {
            $sql .= " WHERE is_active = 1";
        }
        $sql .= " ORDER BY sort_order ASC, price_monthly ASC";
        
        $stmt = $this->pdo->query($sql);
        return $stmt->fetchAll();
    }

    // Find by ID
    public function findById($id)
    {
        $sql = "SELECT * FROM {$this->table} WHERE id = :id LIMIT 1";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':id' => $id]);
        return $stmt->fetch();
    }

    // Find by slug
    public function findBySlug($slug)
    {
        $sql = "SELECT * FROM {$this->table} WHERE slug = :slug LIMIT 1";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':slug' => $slug]);
        return $stmt->fetch();
    }

    // Create plan
    public function create($data)
    {
        $sql = "INSERT INTO {$this->table} 
                (name, slug, description, price_monthly, price_annual, features, max_transactions, max_users, max_organizations, is_active, sort_order) 
                VALUES 
                (:name, :slug, :description, :price_monthly, :price_annual, :features, :max_transactions, :max_users, :max_organizations, :is_active, :sort_order)";
        
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([
            ':name' => $data['name'],
            ':slug' => $data['slug'],
            ':description' => $data['description'] ?? null,
            ':price_monthly' => $data['price_monthly'],
            ':price_annual' => $data['price_annual'],
            ':features' => isset($data['features']) ? json_encode($data['features']) : null,
            ':max_transactions' => $data['max_transactions'] ?? null,
            ':max_users' => $data['max_users'] ?? null,
            ':max_organizations' => $data['max_organizations'] ?? 1,
            ':is_active' => $data['is_active'] ?? 1,
            ':sort_order' => $data['sort_order'] ?? 0
        ]);
    }

    // Update plan
    public function update($id, $data)
    {
        $fields = [];
        $params = [':id' => $id];

        $allowed = ['name', 'slug', 'description', 'price_monthly', 'price_annual', 'features', 
                   'max_transactions', 'max_users', 'max_organizations', 'is_active', 'sort_order'];
        
        foreach ($allowed as $field) {
            if (isset($data[$field])) {
                if ($field === 'features' && is_array($data[$field])) {
                    $fields[] = "$field = :$field";
                    $params[":$field"] = json_encode($data[$field]);
                } else {
                    $fields[] = "$field = :$field";
                    $params[":$field"] = $data[$field];
                }
            }
        }

        if (empty($fields)) {
            return false;
        }

        $sql = "UPDATE {$this->table} SET " . implode(", ", $fields) . " WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute($params);
    }

    // Delete plan
    public function delete($id)
    {
        $sql = "DELETE FROM {$this->table} WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([':id' => $id]);
    }

    // Get price for billing period
    public function getPrice($billingPeriod = 'monthly')
    {
        return $billingPeriod === 'annual' ? $this->price_annual : $this->price_monthly;
    }

    // Parse features JSON
    public function getFeaturesArray()
    {
        if (empty($this->features)) {
            return [];
        }
        $features = json_decode($this->features, true);
        return is_array($features) ? $features : [];
    }
}

