<?php

class Session
{
    public static function start()
    {
        if (session_status() === PHP_SESSION_NONE) {
            require_once __DIR__ . '/Env.php';
            
            // Configure secure session settings from .env
            $cookieHttpOnly = Env::get('SESSION_COOKIE_HTTPONLY', '1') === '1';
            $cookieSecure = (Env::get('SESSION_COOKIE_SECURE', '0') === '1' && self::isHttps()) ? 1 : 0;
            $cookieSameSite = Env::get('SESSION_COOKIE_SAMESITE', 'Strict');
            $sessionTimeout = (int)Env::get('SESSION_TIMEOUT', '1800');
            
            ini_set('session.cookie_httponly', $cookieHttpOnly ? 1 : 0);
            ini_set('session.cookie_secure', $cookieSecure);
            ini_set('session.use_strict_mode', 1);
            ini_set('session.cookie_samesite', $cookieSameSite);
            
            session_start();
            
            // Set session timeout from .env
            if (!self::has('last_activity')) {
                self::set('last_activity', time());
            } else {
                $lastActivity = self::get('last_activity');
                
                if (time() - $lastActivity > $sessionTimeout) {
                    self::destroy();
                    return;
                }
                
                self::set('last_activity', time());
            }
        }
    }
    
    /**
     * Check if request is over HTTPS
     */
    private static function isHttps()
    {
        return (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') 
            || (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https')
            || (!empty($_SERVER['HTTP_X_FORWARDED_SSL']) && $_SERVER['HTTP_X_FORWARDED_SSL'] === 'on')
            || (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443);
    }

    public static function set($key, $value)
    {
        self::start();
        $_SESSION[$key] = $value;
    }

    public static function get($key, $default = null)
    {
        self::start();
        return $_SESSION[$key] ?? $default;
    }

    public static function has($key)
    {
        self::start();
        return isset($_SESSION[$key]);
    }

    public static function remove($key)
    {
        self::start();
        unset($_SESSION[$key]);
    }

    public static function destroy()
    {
        self::start();
        session_destroy();
        $_SESSION = [];
    }

    public static function flash($key, $value = null)
    {
        self::start();
        if ($value === null) {
            $message = $_SESSION['_flash'][$key] ?? null;
            unset($_SESSION['_flash'][$key]);
            return $message;
        }
        $_SESSION['_flash'][$key] = $value;
    }

    public static function setUser($user)
    {
        self::set('user_id', $user['id']);
        self::set('user_name', $user['name']);
        self::set('user_email', $user['email']);
    }

    public static function getUser()
    {
        if (!self::has('user_id')) {
            return null;
        }

        return [
            'id' => self::get('user_id'),
            'name' => self::get('user_name'),
            'email' => self::get('user_email')
        ];
    }

    public static function setOrganization($organization)
    {
        self::set('organization_id', $organization['id']);
        self::set('organization_name', $organization['name']);
        self::set('organization_slug', $organization['slug']);
    }

    public static function getOrganization()
    {
        if (!self::has('organization_id')) {
            return null;
        }

        return [
            'id' => self::get('organization_id'),
            'name' => self::get('organization_name'),
            'slug' => self::get('organization_slug')
        ];
    }

    public static function isAuthenticated()
    {
        return self::has('user_id');
    }
}

