<?php

class WebhookController
{
    private $pdo;
    private $transaction;

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
        $this->transaction = new Transaction($pdo);
    }

    public function handle()
    {
        try {
            // Rate limiting for webhook requests
            $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
            $rateLimitKey = 'webhook_' . $ip;
            
            $webhookAttempts = (int)Env::get('RATE_LIMIT_WEBHOOK_ATTEMPTS', '100');
            $webhookWindow = (int)Env::get('RATE_LIMIT_WEBHOOK_WINDOW', '60');
            
            if (!Security::checkRateLimit($rateLimitKey, $webhookAttempts, $webhookWindow)) {
                http_response_code(429);
                Functions::logError("Webhook rate limit exceeded for IP: $ip");
                die('Rate limit exceeded');
            }
            
            // Validate Telegram secret token header
            $secretToken = $_SERVER['HTTP_X_TELEGRAM_BOT_API_SECRET_TOKEN'] ?? '';
            $expectedToken = Env::get('TELEGRAM_WEBHOOK_SECRET', '');
            
            if (empty($expectedToken)) {
                Functions::logError("TELEGRAM_WEBHOOK_SECRET not configured in .env");
                http_response_code(500);
                die('Webhook configuration error');
            }
            
            if (empty($secretToken) || !hash_equals($expectedToken, $secretToken)) {
                Functions::logError("Invalid webhook secret token from IP: $ip");
                http_response_code(403);
                die('Unauthorized');
            }
            
            // Optional: Validate Telegram IP ranges (basic check)
            // Telegram webhooks come from specific IP ranges: 149.154.160.0/20 and 91.108.4.0/22
            if (Env::get('VALIDATE_TELEGRAM_IP', 'false') === 'true') {
                if (!$this->isTelegramIP($ip)) {
                    Functions::logError("Webhook request from non-Telegram IP: $ip");
                    http_response_code(403);
                    die('Unauthorized');
                }
            }
            
            $data = json_decode(file_get_contents("php://input"), true);

            if (!isset($data['message']['text'])) {
                Functions::logError("No message received.");
                return;
            }

            $message = $data['message']['text'];
            $chatId  = (string)$data['message']['chat']['id'];
            $chatType = isset($data['message']['chat']['type']) ? $data['message']['chat']['type'] : 'private';

            // Find organization by bot token (from webhook URL or message)
            // For now, we'll check all organizations with bot tokens
            $organizationModel = new Organization($this->pdo);
            $organizations = $organizationModel->all();
            
            $matchedOrg = null;
            $telegramAccess = new TelegramAccess($this->pdo);
            
            foreach ($organizations as $org) {
                if (empty($org['telegram_bot_token'])) {
                    continue;
                }
                
                // Check if chat has access to this organization
                if ($telegramAccess->hasAccess($org['id'], $chatId)) {
                    $matchedOrg = $org;
                    break;
                }
            }

            if (!$matchedOrg) {
                Functions::logError("No organization found for chat_id: $chatId");
                TelegramService::sendMessage($chatId, "❌ Access denied. Your chat is not authorized to send payments.");
                return;
            }

            $bank = Parser::detectBank($message);

            if ($bank === "ABA") {
                $parsed = Parser::parseABA($message);
            } elseif ($bank === "RHB") {
                $parsed = Parser::parseRHB($message);
            } else {
                TelegramService::sendMessage($chatId, "❌ Unknown payment format.", $matchedOrg['telegram_bot_token']);
                Functions::logError("Unknown format: " . $message);
                return;
            }

            if (!$parsed || empty($parsed['amount'])) {
                TelegramService::sendMessage($chatId, "⚠️ Failed to extract data.", $matchedOrg['telegram_bot_token']);
                Functions::logError("Parse failed: " . $message);
                return;
            }

            // Add raw message and organization/user info
            $parsed['raw_message'] = $message;
            $parsed['organization_id'] = $matchedOrg['id'];
            $parsed['user_id'] = null; // Could be linked to user if we track Telegram user IDs

            // Save to DB
            $this->transaction->create($parsed);

            // Confirm back to Telegram
            TelegramService::sendMessage($chatId, "✅ Payment recorded: {$parsed['amount']} USD", $matchedOrg['telegram_bot_token']);

        } catch (Exception $e) {
            Functions::logError("WEBHOOK ERROR: " . $e->getMessage());
            http_response_code(500);
        }
    }
    
    /**
     * Check if IP address is from Telegram's IP ranges
     */
    private function isTelegramIP($ip)
    {
        if (!filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            return false;
        }
        
        $ipLong = ip2long($ip);
        
        // Telegram IP ranges:
        // 149.154.160.0/20 (149.154.160.0 - 149.154.175.255)
        // 91.108.4.0/22 (91.108.4.0 - 91.108.7.255)
        
        $range1Start = ip2long('149.154.160.0');
        $range1End = ip2long('149.154.175.255');
        
        $range2Start = ip2long('91.108.4.0');
        $range2End = ip2long('91.108.7.255');
        
        return ($ipLong >= $range1Start && $ipLong <= $range1End) ||
               ($ipLong >= $range2Start && $ipLong <= $range2End);
    }
}
