# Security Documentation

This document outlines all security measures implemented in the Payment Summary System.

## Table of Contents

1. [Authentication & Authorization](#authentication--authorization)
2. [Session Security](#session-security)
3. [CSRF Protection](#csrf-protection)
4. [Input Validation & Sanitization](#input-validation--sanitization)
5. [Output Escaping](#output-escaping)
6. [Webhook Security](#webhook-security)
7. [Rate Limiting](#rate-limiting)
8. [Directory Protection](#directory-protection)
9. [Error Handling](#error-handling)
10. [HTTPS Enforcement](#https-enforcement)
11. [Configuration Security](#configuration-security)
12. [Security Headers](#security-headers)

---

## Authentication & Authorization

### Password Security
- ✅ **Password Hashing**: All passwords are hashed using PHP's `password_hash()` with `PASSWORD_DEFAULT` algorithm (bcrypt)
- ✅ **Password Verification**: Passwords are verified using `password_verify()` to prevent timing attacks
- ✅ **Password Requirements**: Minimum 8 characters required
- ✅ **Session-Based Authentication**: User authentication state stored in secure sessions

### Login Protection
- ✅ **Brute-Force Protection**: Rate limiting implemented (5 attempts per 15 minutes per IP)
- ✅ **Failed Login Logging**: All failed login attempts are logged with IP address
- ✅ **Session Regeneration**: Session ID regenerated after successful login to prevent session fixation
- ✅ **CSRF Protection**: Login form protected with CSRF tokens

### Access Control
- ✅ **Authentication Middleware**: All protected pages require authentication via `Authenticate::handle()`
- ✅ **Protected Routes**: 
  - `/public/summary.php`
  - `/public/transaction-list.php`
  - `/public/index.php` (dashboard)
  - `/public/admin-users.php`
  - `/public/admin-plans.php`
  - `/public/settings.php`
- ✅ **Guest Routes**: Login and registration pages redirect authenticated users

---

## Session Security

### Session Configuration
- ✅ **HttpOnly Cookies**: Session cookies set with `HttpOnly` flag to prevent XSS access
- ✅ **Secure Cookies**: Cookies set with `Secure` flag when HTTPS is detected
- ✅ **SameSite Protection**: Cookies set with `SameSite=Strict` to prevent CSRF
- ✅ **Strict Mode**: PHP session strict mode enabled
- ✅ **Session Timeout**: 30-minute inactivity timeout

### Session Management
- ✅ **Session Regeneration**: Session ID regenerated after login
- ✅ **Session Fixation Prevention**: Old session destroyed when regenerating ID
- ✅ **IP Tracking**: User IP address stored in session for validation (optional)
- ✅ **User Agent Tracking**: User agent stored in session for validation

---

## CSRF Protection

### Implementation
- ✅ **CSRF Tokens**: All POST forms include CSRF tokens
- ✅ **Token Generation**: Cryptographically secure random tokens (32 bytes, hex encoded)
- ✅ **Token Validation**: All POST requests validated before processing
- ✅ **Secure Comparison**: Uses `hash_equals()` for constant-time token comparison

### Protected Forms
- ✅ Login form (`/public/login.php`)
- ✅ Registration form (`/public/register.php`)
- ✅ User management forms (`/public/admin-users.php`)
- ✅ Plan management forms (`/public/admin-plans.php`)
- ✅ Settings forms (`/public/settings.php`)
- ✅ All admin action forms (delete, update, toggle, etc.)

---

## Input Validation & Sanitization

### Validation Methods
- ✅ **Email Validation**: `filter_var()` with `FILTER_VALIDATE_EMAIL`
- ✅ **Integer Validation**: Type checking with min/max bounds
- ✅ **String Sanitization**: `htmlspecialchars()` with `ENT_QUOTES` and `UTF-8`
- ✅ **Input Trimming**: All string inputs trimmed of whitespace
- ✅ **Length Limits**: Maximum length validation for string fields

### Sanitization
- ✅ **XSS Prevention**: All user inputs sanitized before storage
- ✅ **SQL Injection Prevention**: Prepared statements used for all database queries
- ✅ **Directory Traversal Prevention**: File path validation using `realpath()`
- ✅ **Type Casting**: Proper type casting for numeric inputs

### Validated Inputs
- User names, emails, passwords
- Organization names, slugs
- Transaction data
- Telegram chat IDs
- All form submissions

---

## Output Escaping

### Implementation
- ✅ **HTML Escaping**: All user-generated content escaped with `htmlspecialchars()`
- ✅ **Context-Aware Escaping**: Proper escaping for HTML attributes, text content, and URLs
- ✅ **Consistent Usage**: `Security::escape()` helper used throughout views

### Escaped Outputs
- ✅ Transaction lists (payer names, merchants, amounts)
- ✅ User names and emails in admin panels
- ✅ Organization names
- ✅ All table data
- ✅ Form values
- ✅ Error and success messages

---

## Webhook Security

### Telegram Webhook Protection
- ✅ **Secret Token Validation**: Validates `X-Telegram-Bot-Api-Secret-Token` header
- ✅ **Token Comparison**: Uses `hash_equals()` for constant-time comparison
- ✅ **Missing Token Rejection**: All requests without valid token are rejected
- ✅ **Invalid Attempt Logging**: Failed webhook attempts logged with IP address
- ✅ **IP Range Validation**: Optional validation of Telegram IP ranges (149.154.160.0/20, 91.108.4.0/22)

### Rate Limiting
- ✅ **Webhook Rate Limiting**: 100 requests per minute per IP address
- ✅ **Lockout Period**: Temporary lockout after rate limit exceeded

### Access Control
- ✅ **Chat ID Validation**: Only authorized chat IDs can send transactions
- ✅ **Organization Matching**: Transactions matched to correct organization
- ✅ **Access Logging**: All webhook access attempts logged

---

## Rate Limiting

### Implementation
- ✅ **File-Based Storage**: Rate limit data persisted in files (survives server restarts)
- ✅ **Per-IP Tracking**: Rate limits tracked by IP address
- ✅ **Configurable Limits**: Limits configurable per endpoint

### Protected Endpoints
- ✅ **Login**: 5 attempts per 15 minutes
- ✅ **Webhook**: 100 requests per minute
- ✅ **Lockout Period**: Automatic lockout after limit exceeded

### Rate Limit Storage
- Location: `/storage/cache/rate_limit_*.json`
- Automatic cleanup: Expired windows automatically reset
- Thread-safe: File locking prevents race conditions

---

## Directory Protection

### Protected Directories
All backend directories are protected with `.htaccess` files:

- ✅ `/app/` - Application code (deny all)
- ✅ `/database/` - Database migrations and seeders (deny all)
- ✅ `/storage/` - Logs and cache (deny all)
- ✅ `/tests/` - Test files (deny all)

### File Protection
- ✅ `.env` file protected from web access
- ✅ Directory listing disabled (`Options -Indexes`)
- ✅ Sensitive files cannot be accessed via browser

---

## Error Handling

### Production Mode
- ✅ **Error Display**: Errors hidden from users in production
- ✅ **Generic Messages**: Generic error messages shown to users
- ✅ **Detailed Logging**: All errors logged to `/storage/logs/system.log`
- ✅ **Stack Traces**: Full stack traces logged but not displayed

### Error Logging
- ✅ **Centralized Logging**: All errors logged to single file
- ✅ **Timestamped Logs**: All log entries include timestamps
- ✅ **Error Context**: Logs include file, line number, and stack trace
- ✅ **Exception Handling**: Uncaught exceptions handled gracefully

### Log File Security
- ✅ **Protected Location**: Logs stored in protected `/storage/logs/` directory
- ✅ **Not Web Accessible**: Logs cannot be accessed via browser
- ✅ **Regular Monitoring**: Logs should be monitored for security issues

---

## HTTPS Enforcement

### Implementation
- ✅ **HTTPS Redirect**: Redirect configured in `.htaccess` (commented for development)
- ✅ **Secure Cookies**: Cookies automatically set to secure when HTTPS detected
- ✅ **Protocol Detection**: Detects HTTPS via multiple headers (X-Forwarded-Proto, etc.)

### Configuration
- Uncomment HTTPS redirect in `.htaccess` for production:
```apache
RewriteCond %{HTTPS} off
RewriteCond %{HTTP:X-Forwarded-Proto} !https
RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
```

---

## Configuration Security

### Environment Variables
- ✅ **`.env` File**: All sensitive configuration in `.env` file
- ✅ **File Protection**: `.env` file protected from web access
- ✅ **Example File**: `.env.example` provided (no secrets)
- ✅ **Git Ignore**: `.env` should be in `.gitignore`

### Secure Configuration
- ✅ **Database Credentials**: Stored in `.env`, not in code
- ✅ **Bot Tokens**: Telegram bot tokens in `.env`
- ✅ **Webhook Secrets**: Secret tokens in `.env`
- ✅ **No Hardcoded Secrets**: No secrets hardcoded in source code

### Required Environment Variables
- `DB_HOST`, `DB_NAME`, `DB_USER`, `DB_PASS`
- `TELEGRAM_BOT_TOKEN`
- `TELEGRAM_WEBHOOK_SECRET`
- `APP_ENV` (production/development)
- `APP_DEBUG` (true/false)

---

## Security Headers

### HTTP Security Headers
All security headers are set automatically via `Security::setSecurityHeaders()`:

- ✅ **X-Frame-Options**: `SAMEORIGIN` (prevents clickjacking)
- ✅ **X-Content-Type-Options**: `nosniff` (prevents MIME sniffing)
- ✅ **X-XSS-Protection**: `1; mode=block` (XSS protection)
- ✅ **Referrer-Policy**: `strict-origin-when-cross-origin`
- ✅ **Content-Security-Policy**: Basic CSP configured

### Header Configuration
Headers are set in `routes/web.php` and applied to all requests.

---

## Security Best Practices

### For Administrators

1. **Regular Updates**
   - Keep PHP updated to latest stable version
   - Keep server software updated
   - Monitor security advisories

2. **Password Management**
   - Use strong, unique passwords
   - Change default passwords immediately
   - Consider password rotation policies

3. **Monitoring**
   - Regularly review `/storage/logs/system.log`
   - Monitor for failed login attempts
   - Check for suspicious webhook activity

4. **Backup Security**
   - Encrypt database backups
   - Secure backup storage locations
   - Test backup restoration regularly

5. **Access Control**
   - Limit admin user accounts
   - Use principle of least privilege
   - Regularly audit user access

### For Developers

1. **Code Review**
   - Review all code changes for security issues
   - Test security features after changes
   - Follow secure coding practices

2. **Dependencies**
   - Keep dependencies updated
   - Review dependency security advisories
   - Use only trusted packages

3. **Testing**
   - Test authentication flows
   - Test CSRF protection
   - Test input validation
   - Test error handling

---

## Security Checklist

### Pre-Deployment
- [ ] All environment variables configured in `.env`
- [ ] `.env` file permissions set to 600 (owner read/write only)
- [ ] HTTPS redirect enabled in `.htaccess`
- [ ] `APP_ENV` set to `production`
- [ ] `APP_DEBUG` set to `false`
- [ ] Strong `TELEGRAM_WEBHOOK_SECRET` generated
- [ ] Database credentials are secure
- [ ] All default passwords changed
- [ ] Directory permissions reviewed
- [ ] Log file permissions secure

### Post-Deployment
- [ ] Test login functionality
- [ ] Test CSRF protection
- [ ] Test webhook security
- [ ] Verify error pages don't leak information
- [ ] Check security headers are present
- [ ] Verify HTTPS is enforced
- [ ] Test rate limiting
- [ ] Review initial log entries

---

## Incident Response

### If Security Breach Suspected

1. **Immediate Actions**
   - Change all passwords
   - Rotate API tokens and secrets
   - Review recent log entries
   - Check for unauthorized access

2. **Investigation**
   - Review `/storage/logs/system.log`
   - Check server access logs
   - Review database for unauthorized changes
   - Check for suspicious transactions

3. **Recovery**
   - Restore from clean backup if needed
   - Update all credentials
   - Review and fix security vulnerabilities
   - Notify affected users if necessary

---

## Security Contact

For security issues or questions:
- Review logs: `/storage/logs/system.log`
- Check documentation: `CONFIGURATION.md` and `PRODUCTION_DEVELOPMENT.md`
- Review code: Security implementations in `/app/Helpers/Security.php`

## Related Documentation

- [README.md](README.md) - Project overview and installation
- [CONFIGURATION.md](CONFIGURATION.md) - Configuration guide
- [PRODUCTION_DEVELOPMENT.md](PRODUCTION_DEVELOPMENT.md) - Production deployment guide

---

**Last Updated**: 2024
**Version**: 1.0

