# Production & Development Setup Guide

This guide covers how to set up and run the Payment Summary System in both development and production environments.

## 🔧 Development Environment Setup

### Local Development (Laragon/XAMPP/WAMP)

#### 1. Environment Configuration

Create a `.env` file in the project root:

```env
# Database Configuration
DB_HOST=localhost
DB_NAME=sumtel
DB_USER=root
DB_PASS=

# Application Configuration
APP_NAME=Payment Summary System
APP_URL=http://localhost/payments/public
APP_TIMEZONE=Asia/Phnom_Penh
APP_ENV=development
APP_DEBUG=true

# Telegram Bot Configuration
TELEGRAM_BOT_TOKEN=YOUR_TELEGRAM_BOT_TOKEN_HERE
```

#### 2. Database Setup

**Using phpMyAdmin:**
1. Open `http://localhost/phpmyadmin`
2. Create database: `sumtel`
3. Import: `database/migrate_all.sql`

**Using Command Line:**
```bash
mysql -u root -p
CREATE DATABASE sumtel;
USE sumtel;
SOURCE database/migrate_all.sql;
```

**Using PHP Script:**
```bash
php database/migrate.php
```

#### 3. Web Server Configuration

**Laragon:**
- Project path: `C:\laragon\www\payments`
- Access URL: `http://localhost/payments/public/` or `http://payments.test/`
- Document root should point to `public/` folder

**XAMPP:**
- Project path: `C:\xampp\htdocs\payments`
- Access URL: `http://localhost/payments/public/`

#### 4. Testing Telegram Webhook Locally

Since Telegram requires a public URL, use **ngrok**:

1. **Install ngrok:** https://ngrok.com/download
2. **Start ngrok:**
   ```bash
   ngrok http 80
   ```
3. **Copy HTTPS URL** (e.g., `https://abc123.ngrok.io`)
4. **Set webhook:**
   ```
   https://api.telegram.org/botYOUR_TOKEN/setWebhook?url=https://abc123.ngrok.io/payments/public/webhook.php
   ```

**Note:** ngrok URLs change on each restart. Update webhook when restarting ngrok.

#### 5. Development Best Practices

- **Enable Debug Mode:** Set `APP_DEBUG=true` in `.env`
- **Check Logs:** Monitor `storage/logs/system.log` for errors
- **Test Parser:** Use `php tests/parser-test.php` to test message formats
- **Use Install Checker:** Visit `public/install-check.php` to verify setup

## 🚀 Production Environment Setup

### 1. Server Requirements

- **PHP:** 7.4 or higher (8.3+ recommended)
- **MySQL:** 5.7 or higher (or MariaDB 10.3+)
- **Web Server:** Apache 2.4+ or Nginx 1.18+
- **SSL Certificate:** Required for Telegram webhooks (HTTPS)
- **PHP Extensions:** PDO, PDO_MySQL, JSON, mbstring, curl

### 2. Production Environment Configuration

Create `.env` file with production settings:

```env
# Database Configuration
DB_HOST=localhost
DB_NAME=your_production_db
DB_USER=your_db_user
DB_PASS=your_secure_password

# Application Configuration
APP_NAME=Payment Summary System
APP_URL=https://yourdomain.com/public
APP_TIMEZONE=Asia/Phnom_Penh
APP_ENV=production
APP_DEBUG=false

# Telegram Bot Configuration
TELEGRAM_BOT_TOKEN=your_production_bot_token
```

**Important Production Settings:**
- `APP_ENV=production` - Disables debug mode
- `APP_DEBUG=false` - Hides error details from users
- Use strong database passwords
- Use HTTPS URLs only

### 3. Database Setup

**Option 1: Direct Import**
```bash
mysql -u your_user -p your_database < database/migrate_all.sql
```

**Option 2: Using phpMyAdmin**
1. Create database
2. Import `database/migrate_all.sql`

**Option 3: PHP Script**
```bash
php database/migrate.php
```

### 4. File Permissions

**Linux/Mac:**
```bash
# Set proper permissions
chmod -R 755 storage/
chmod -R 755 storage/logs/
chmod -R 755 storage/cache/
chmod 644 .env

# Set ownership (replace www-data with your web server user)
chown -R www-data:www-data storage/
chown -R www-data:www-data public/
```

**Windows:**
- Ensure IIS_IUSRS or your web server user has read/write access to:
  - `storage/` directory
  - `.env` file

### 5. Web Server Configuration

#### Apache Configuration

**Virtual Host Example:**
```apache
<VirtualHost *:80>
    ServerName yourdomain.com
    ServerAlias www.yourdomain.com
    
    DocumentRoot /var/www/payments/public
    
    <Directory /var/www/payments/public>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>
    
    ErrorLog ${APACHE_LOG_DIR}/payments_error.log
    CustomLog ${APACHE_LOG_DIR}/payments_access.log combined
</VirtualHost>
```

**Enable SSL (Required for Telegram):**
```apache
<VirtualHost *:443>
    ServerName yourdomain.com
    
    SSLEngine on
    SSLCertificateFile /path/to/certificate.crt
    SSLCertificateKeyFile /path/to/private.key
    
    DocumentRoot /var/www/payments/public
    
    <Directory /var/www/payments/public>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>
</VirtualHost>
```

#### Nginx Configuration

```nginx
server {
    listen 80;
    server_name yourdomain.com;
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    server_name yourdomain.com;
    
    ssl_certificate /path/to/certificate.crt;
    ssl_certificate_key /path/to/private.key;
    
    root /var/www/payments/public;
    index index.php;
    
    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }
    
    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.3-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
        include fastcgi_params;
    }
    
    location ~ /\. {
        deny all;
    }
}
```

### 6. Security Hardening

#### 1. Protect `.env` File

**Apache (.htaccess in root):**
```apache
<Files ".env">
    Order allow,deny
    Deny from all
</Files>
```

**Nginx:**
```nginx
location ~ /\.env {
    deny all;
    return 404;
}
```

#### 2. Disable Directory Listing

**Apache (.htaccess in public):**
```apache
Options -Indexes
```

#### 3. Hide PHP Version
```php
// In php.ini
expose_php = Off
```

#### 4. Security Headers

Security headers are automatically set via `Security::setSecurityHeaders()` in `routes/web.php`. The following headers are configured:
- `X-Content-Type-Options: nosniff`
- `X-Frame-Options: SAMEORIGIN`
- `X-XSS-Protection: 1; mode=block`
- `Referrer-Policy: strict-origin-when-cross-origin`
- `Content-Security-Policy` (basic CSP)

**Note**: All security features (CSRF protection, rate limiting, security headers, input validation) are already implemented. See [SECURITY.md](SECURITY.md) for complete documentation.

#### 5. Database Security
- Use strong passwords
- Create dedicated database user with minimal privileges
- Restrict database access to localhost only
- Regular backups

### 7. Telegram Webhook Setup

1. **Set Production Webhook:**
   ```
   https://api.telegram.org/botYOUR_TOKEN/setWebhook?url=https://yourdomain.com/public/webhook.php
   ```

2. **Verify Webhook:**
   ```
   https://api.telegram.org/botYOUR_TOKEN/getWebhookInfo
   ```

3. **Test Webhook:**
   Send a test message to your bot and check if it's received

### 8. Performance Optimization

#### 1. Enable OPcache (PHP)
```ini
; In php.ini
opcache.enable=1
opcache.memory_consumption=128
opcache.max_accelerated_files=10000
opcache.validate_timestamps=0  ; Set to 0 in production
```

#### 2. Enable Gzip Compression

**Apache (.htaccess):**
```apache
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css text/javascript application/javascript application/json
</IfModule>
```

**Nginx:**
```nginx
gzip on;
gzip_types text/plain text/css application/json application/javascript text/xml application/xml;
```

#### 3. Browser Caching

**Apache (.htaccess in public):**
```apache
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType image/jpg "access plus 1 year"
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/gif "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
</IfModule>
```

### 9. Monitoring & Logging

#### 1. Application Logs
- Location: `storage/logs/system.log`
- Monitor for errors regularly
- Set up log rotation

#### 2. Server Logs
- Apache: `/var/log/apache2/error.log`
- Nginx: `/var/log/nginx/error.log`
- Monitor for 500 errors, slow requests

#### 3. Database Monitoring
- Monitor slow queries
- Check connection pool usage
- Regular backup verification

### 10. Backup Strategy

#### Database Backups
```bash
# Daily backup script
mysqldump -u user -p database_name > backup_$(date +%Y%m%d).sql

# Automated backup (cron)
0 2 * * * mysqldump -u user -p database_name > /backups/db_$(date +\%Y\%m\%d).sql
```

#### File Backups
- Backup `.env` file (securely)
- Backup `storage/logs/` if needed
- Version control for code

### 11. Deployment Checklist

- [ ] Server meets requirements (PHP, MySQL, Web Server)
- [ ] SSL certificate installed and configured
- [ ] Database created and migrated
- [ ] `.env` file configured with production values
- [ ] `APP_DEBUG=false` in production
- [ ] File permissions set correctly
- [ ] Web server configured (Apache/Nginx)
- [ ] `.env` file protected from web access
- [ ] Telegram webhook configured
- [ ] Webhook tested and working
- [ ] Logs directory writable
- [ ] Backup strategy in place
- [ ] Monitoring set up
- [ ] Security headers configured
- [ ] Performance optimizations enabled

### 12. Troubleshooting Production Issues

#### Database Connection Issues
- Check database credentials in `.env`
- Verify database server is accessible
- Check firewall rules
- Review database server logs

#### Webhook Not Working
- Verify HTTPS is working
- Check webhook URL is accessible
- Review `storage/logs/system.log`
- Test webhook endpoint directly
- Verify bot token is correct

#### Performance Issues
- Enable OPcache
- Check slow query log
- Review server resources (CPU, RAM)
- Optimize database indexes
- Enable caching where possible

#### Security Issues
- Review access logs for suspicious activity
- Check for unauthorized access attempts
- Verify `.env` file is protected
- Review application logs for errors
- Keep PHP and server software updated

## 🔄 Development vs Production Differences

| Setting | Development | Production |
|---------|------------|------------|
| `APP_ENV` | `development` | `production` |
| `APP_DEBUG` | `true` | `false` |
| `APP_URL` | `http://localhost/...` | `https://yourdomain.com/...` |
| Error Display | Full errors | Generic messages |
| Logging | Verbose | Essential only |
| SSL | Not required | Required |
| Webhook URL | ngrok URL | Production domain |

## 📝 Environment Variables Reference

| Variable | Description | Example |
|----------|-------------|---------|
| `DB_HOST` | Database host | `localhost` |
| `DB_NAME` | Database name | `sumtel` |
| `DB_USER` | Database username | `root` |
| `DB_PASS` | Database password | `your_password` |
| `APP_NAME` | Application name | `Payment Summary System` |
| `APP_URL` | Base URL | `https://yourdomain.com/public` |
| `APP_TIMEZONE` | Timezone | `Asia/Phnom_Penh` |
| `APP_ENV` | Environment | `production` or `development` |
| `APP_DEBUG` | Debug mode | `true` or `false` |
| `TELEGRAM_BOT_TOKEN` | Telegram bot token | `123456:ABC-DEF...` |

## 🆘 Support

For production issues:
1. Check `storage/logs/system.log`
2. Review server error logs
3. Verify all configuration settings
4. Test webhook connectivity
5. Check database connectivity

---

**Remember:** Always test changes in development before deploying to production!
