# Environment Configuration Enhancements

This document outlines all the enhancements made to move configuration from hardcoded values to `.env` file.

## ✅ Changes Made

### 1. Database Configuration (`app/Config/db.php`)

**Before:**
- Hardcoded charset: `utf8`
- Hardcoded database name default: `seekspgq_sumtel`

**After:**
- ✅ `DB_CHARSET` from `.env` (default: `utf8mb4`)
- ✅ Database name from `.env` (default: `sumtel`)
- ✅ Added `PDO::ATTR_EMULATE_PREPARES => false` for better security

**New Environment Variables:**
```env
DB_CHARSET="utf8mb4"
```

---

### 2. Session Configuration (`app/Helpers/Session.php`)

**Before:**
- Hardcoded session timeout: `1800` seconds (30 minutes)
- Hardcoded cookie settings

**After:**
- ✅ `SESSION_TIMEOUT` from `.env` (default: `1800`)
- ✅ `SESSION_COOKIE_HTTPONLY` from `.env` (default: `1`)
- ✅ `SESSION_COOKIE_SECURE` from `.env` (default: `0` for dev, `1` for prod)
- ✅ `SESSION_COOKIE_SAMESITE` from `.env` (default: `Strict`)

**New Environment Variables:**
```env
SESSION_TIMEOUT="1800"
SESSION_COOKIE_HTTPONLY="1"
SESSION_COOKIE_SECURE="0"
SESSION_COOKIE_SAMESITE="Strict"
```

---

### 3. Rate Limiting (`app/Helpers/Security.php`)

**Before:**
- Hardcoded default rate limit: `5 attempts per 300 seconds`

**After:**
- ✅ Default rate limit attempts from `.env`: `RATE_LIMIT_DEFAULT_ATTEMPTS`
- ✅ Default rate limit window from `.env`: `RATE_LIMIT_DEFAULT_WINDOW`
- ✅ Can be overridden per endpoint

**New Environment Variables:**
```env
RATE_LIMIT_DEFAULT_ATTEMPTS="5"
RATE_LIMIT_DEFAULT_WINDOW="300"
```

---

### 4. Login Rate Limiting (`app/Auth/AuthController.php`)

**Before:**
- Hardcoded: `5 attempts per 900 seconds (15 minutes)`

**After:**
- ✅ `RATE_LIMIT_LOGIN_ATTEMPTS` from `.env` (default: `5`)
- ✅ `RATE_LIMIT_LOGIN_WINDOW` from `.env` (default: `900`)
- ✅ Dynamic error message with actual timeout duration

**New Environment Variables:**
```env
RATE_LIMIT_LOGIN_ATTEMPTS="5"
RATE_LIMIT_LOGIN_WINDOW="900"
```

---

### 5. Registration Rate Limiting (`app/Auth/AuthController.php`)

**Before:**
- ❌ No rate limiting on registration

**After:**
- ✅ `RATE_LIMIT_REGISTER_ATTEMPTS` from `.env` (default: `3`)
- ✅ `RATE_LIMIT_REGISTER_WINDOW` from `.env` (default: `3600` = 1 hour)
- ✅ Rate limiting now implemented for registration

**New Environment Variables:**
```env
RATE_LIMIT_REGISTER_ATTEMPTS="3"
RATE_LIMIT_REGISTER_WINDOW="3600"
```

---

### 6. Webhook Rate Limiting (`app/Controllers/WebhookController.php`)

**Before:**
- Hardcoded: `100 requests per 60 seconds`

**After:**
- ✅ `RATE_LIMIT_WEBHOOK_ATTEMPTS` from `.env` (default: `100`)
- ✅ `RATE_LIMIT_WEBHOOK_WINDOW` from `.env` (default: `60`)

**New Environment Variables:**
```env
RATE_LIMIT_WEBHOOK_ATTEMPTS="100"
RATE_LIMIT_WEBHOOK_WINDOW="60"
```

---

### 7. Content Security Policy (`app/Helpers/Security.php`)

**Before:**
- Hardcoded CSP policy string

**After:**
- ✅ `CSP_POLICY` from `.env` (optional, uses default if empty)
- ✅ Allows customization of CSP without code changes

**New Environment Variables:**
```env
CSP_POLICY=""
# Leave empty for default, or customize:
# CSP_POLICY="default-src 'self'; script-src 'self' 'unsafe-inline' https://cdn.tailwindcss.com; ..."
```

---

### 8. Timezone Configuration (`routes/web.php`)

**Before:**
- Timezone not automatically set from config

**After:**
- ✅ Timezone automatically set from `APP_TIMEZONE` in `.env`
- ✅ Uses `date_default_timezone_set()` on application start

**Existing Environment Variable (now properly used):**
```env
APP_TIMEZONE="Asia/Phnom_Penh"
```

---

## 📋 Complete Environment Variables List

### Application Configuration
```env
APP_NAME="Payment Summary System"
APP_URL="https://yourdomain.com/payments/public"
APP_TIMEZONE="Asia/Phnom_Penh"
APP_ENV="production"
APP_DEBUG="false"
```

### Database Configuration
```env
DB_HOST="localhost"
DB_NAME="your_database_name"
DB_USER="your_database_user"
DB_PASS="your_secure_password"
DB_CHARSET="utf8mb4"
```

### Telegram Configuration
```env
TELEGRAM_BOT_TOKEN="your_telegram_bot_token"
TELEGRAM_WEBHOOK_SECRET="your_webhook_secret"
VALIDATE_TELEGRAM_IP="false"
```

### Rate Limiting Configuration
```env
# Default rate limits (used when not specified)
RATE_LIMIT_DEFAULT_ATTEMPTS="5"
RATE_LIMIT_DEFAULT_WINDOW="300"

# Login rate limits
RATE_LIMIT_LOGIN_ATTEMPTS="5"
RATE_LIMIT_LOGIN_WINDOW="900"

# Registration rate limits
RATE_LIMIT_REGISTER_ATTEMPTS="3"
RATE_LIMIT_REGISTER_WINDOW="3600"

# Webhook rate limits
RATE_LIMIT_WEBHOOK_ATTEMPTS="100"
RATE_LIMIT_WEBHOOK_WINDOW="60"
```

### Session Configuration
```env
SESSION_TIMEOUT="1800"
SESSION_COOKIE_HTTPONLY="1"
SESSION_COOKIE_SECURE="1"
SESSION_COOKIE_SAMESITE="Strict"
```

### Security Configuration
```env
CSP_POLICY=""
```

---

## 🎯 Benefits

1. **Flexibility**: All configuration can be changed without modifying code
2. **Environment-Specific**: Different settings for development, staging, and production
3. **Security**: Sensitive values kept out of code
4. **Maintainability**: Single source of truth for configuration
5. **Deployment**: Easy to configure for different hosting environments (cPanel, VPS, etc.)

---

## 📝 Migration Guide

### For Existing Installations

1. **Update your `.env` file** with new variables:
   ```env
   # Add these new variables
   DB_CHARSET="utf8mb4"
   
   RATE_LIMIT_LOGIN_ATTEMPTS="5"
   RATE_LIMIT_LOGIN_WINDOW="900"
   RATE_LIMIT_REGISTER_ATTEMPTS="3"
   RATE_LIMIT_REGISTER_WINDOW="3600"
   RATE_LIMIT_WEBHOOK_ATTEMPTS="100"
   RATE_LIMIT_WEBHOOK_WINDOW="60"
   
   SESSION_TIMEOUT="1800"
   SESSION_COOKIE_HTTPONLY="1"
   SESSION_COOKIE_SECURE="1"
   SESSION_COOKIE_SAMESITE="Strict"
   
   CSP_POLICY=""
   ```

2. **Optional**: Customize values based on your needs
   - Adjust rate limits for your traffic patterns
   - Modify session timeout for your security requirements
   - Customize CSP policy if needed

3. **Test**: Verify all functionality works with new configuration

---

## 🔍 Verification

To verify all configuration is loaded from `.env`:

1. **Check logs**: No hardcoded values should appear in error logs
2. **Test rate limiting**: Try exceeding limits to verify they use `.env` values
3. **Check session**: Verify session timeout matches `.env` setting
4. **Database**: Verify charset is `utf8mb4` (or your configured value)

---

## 📚 Related Files

- `.env.example` - Development template
- `env.production.example` - Production template
- `app/Helpers/Env.php` - Environment variable loader
- `app/Config/app.php` - Application configuration
- `app/Config/db.php` - Database configuration

---

**Last Updated**: 2024  
**Version**: 2.0
