# Configuration Guide

This document provides step-by-step instructions for configuring the Payment Summary System for production deployment.

## Table of Contents

1. [Initial Setup](#initial-setup)
2. [Environment Configuration](#environment-configuration)
3. [Database Configuration](#database-configuration)
4. [Telegram Bot Setup](#telegram-bot-setup)
5. [Webhook Configuration](#webhook-configuration)
6. [Security Configuration](#security-configuration)
7. [Production Deployment](#production-deployment)
8. [Troubleshooting](#troubleshooting)

---

## Initial Setup

### 1. File Permissions

Set proper file permissions:

```bash
# Set directory permissions
chmod 755 /path/to/payments
chmod 755 /path/to/payments/public
chmod 755 /path/to/payments/storage
chmod 755 /path/to/payments/storage/logs
chmod 755 /path/to/payments/storage/cache

# Set file permissions
chmod 644 /path/to/payments/.htaccess
chmod 600 /path/to/payments/.env  # IMPORTANT: Only owner can read/write
```

### 2. Create Environment File

Copy the example environment file:

```bash
cp .env.example .env
```

---

## Environment Configuration

### 1. Edit `.env` File

Open `.env` file and configure all variables:

```env
# Application Configuration
APP_NAME="Payment Summary System"
APP_URL="https://yourdomain.com/public"
APP_TIMEZONE="Asia/Phnom_Penh"
APP_ENV="production"
APP_DEBUG="false"

# Database Configuration
DB_HOST="localhost"
DB_NAME="your_database_name"
DB_USER="your_database_user"
DB_PASS="your_secure_database_password"

# Telegram Bot Configuration
TELEGRAM_BOT_TOKEN="your_telegram_bot_token_here"
TELEGRAM_WEBHOOK_SECRET="your_webhook_secret_token_here"

# Security Settings
VALIDATE_TELEGRAM_IP="false"
```

### 2. Generate Webhook Secret

Generate a secure random secret for webhook validation:

**Using OpenSSL:**
```bash
openssl rand -hex 32
```

**Using PHP:**
```php
<?php
echo bin2hex(random_bytes(32));
?>
```

Copy the generated value to `TELEGRAM_WEBHOOK_SECRET` in `.env`.

### 3. Application Settings

#### APP_NAME
- Your application name (displayed in UI)
- Example: `"Payment Summary System"`

#### APP_URL
- Full URL to your public directory
- Must include protocol (https://)
- Example: `"https://payments.yourdomain.com/public"`

#### APP_TIMEZONE
- PHP timezone identifier
- Example: `"Asia/Phnom_Penh"`, `"UTC"`, `"America/New_York"`

#### APP_ENV
- Environment mode
- Options: `"production"` or `"development"`
- **Must be `"production"` for live deployment**

#### APP_DEBUG
- Enable/disable debug mode
- Options: `"true"` or `"false"`
- **Must be `"false"` for production**

---

## Database Configuration

### 1. Create Database

Create a new MySQL database:

```sql
CREATE DATABASE your_database_name CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

### 2. Create Database User

Create a dedicated database user:

```sql
CREATE USER 'your_database_user'@'localhost' IDENTIFIED BY 'your_secure_database_password';
GRANT ALL PRIVILEGES ON your_database_name.* TO 'your_database_user'@'localhost';
FLUSH PRIVILEGES;
```

### 3. Configure Database Connection

Update `.env` with database credentials:

```env
DB_HOST="localhost"
DB_NAME="your_database_name"
DB_USER="your_database_user"
DB_PASS="your_secure_database_password"
```

### 4. Run Migrations

Execute database migrations:

```bash
php database/migrate.php
```

Or manually import SQL files from `/database/migrations/` in order.

### 5. Create Admin User

Create your first admin user:

**Option 1: Using Seeder**
```bash
# Import seed file
mysql -u your_database_user -p your_database_name < database/seeders/seed_superadmin.sql
```

**Option 2: Manual SQL**
```sql
INSERT INTO users (name, email, password_hash, created_at) 
VALUES (
    'Admin User',
    'admin@yourdomain.com',
    '$2y$10$YourHashedPasswordHere',
    NOW()
);
```

To generate password hash:
```php
<?php
echo password_hash('your_password', PASSWORD_DEFAULT);
?>
```

---

## Telegram Bot Setup

### 1. Create Telegram Bot

1. Open Telegram and search for `@BotFather`
2. Send `/newbot` command
3. Follow instructions to create your bot
4. Copy the bot token provided

### 2. Configure Bot Token

Add bot token to `.env`:

```env
TELEGRAM_BOT_TOKEN="123456789:ABCdefGHIjklMNOpqrsTUVwxyz"
```

### 3. Get Chat IDs

To allow users/groups to send payments:

**For Personal Chat:**
1. Forward a message from the user to `@userinfobot`
2. Copy the Chat ID (numeric value)

**For Groups:**
1. Add `@userinfobot` to the group
2. Send any message in the group
3. Bot will reply with group Chat ID (negative number)

### 4. Add Telegram Access

1. Log in to admin panel
2. Go to Settings
3. Add Telegram Chat ID in "Telegram Access Control" section
4. Save settings

---

## Webhook Configuration

### 1. Generate Webhook Secret

Already done in [Environment Configuration](#environment-configuration) step 2.

### 2. Configure Webhook Secret in Telegram

When setting webhook URL in Telegram Bot API, include the secret token:

**Using cURL:**
```bash
curl -X POST "https://api.telegram.org/bot<YOUR_BOT_TOKEN>/setWebhook" \
  -H "Content-Type: application/json" \
  -d '{
    "url": "https://yourdomain.com/public/webhook.php",
    "secret_token": "your_webhook_secret_from_env"
  }'
```

**Using Admin Panel:**
1. Log in to admin panel
2. Go to "Manage Users"
3. Find your organization
4. Click "Set Webhook" button
5. Webhook will be configured automatically

### 3. Verify Webhook

Check webhook status:

```bash
curl "https://api.telegram.org/bot<YOUR_BOT_TOKEN>/getWebhookInfo"
```

### 4. Test Webhook

Send a test payment message to your bot and verify it appears in the system.

---

## Security Configuration

### 1. Enable HTTPS

Uncomment HTTPS redirect in `.htaccess`:

```apache
# Force HTTPS in production
RewriteCond %{HTTPS} off
RewriteCond %{HTTP:X-Forwarded-Proto} !https
RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
```

### 2. Set File Permissions

```bash
# .env file - owner read/write only
chmod 600 .env

# Storage directories - writable by web server
chmod 755 storage
chmod 755 storage/logs
chmod 755 storage/cache
```

### 3. Enable Telegram IP Validation (Optional)

For additional security, enable Telegram IP validation:

```env
VALIDATE_TELEGRAM_IP="true"
```

**Note**: This may cause issues if your server is behind a proxy. Test thoroughly.

### 4. Review Security Headers

Security headers are automatically set via `Security::setSecurityHeaders()` in `routes/web.php`. Verify they're present:

```bash
curl -I https://yourdomain.com/public/
```

Look for:
- `X-Frame-Options: SAMEORIGIN`
- `X-Content-Type-Options: nosniff`
- `X-XSS-Protection: 1; mode=block`
- `Referrer-Policy: strict-origin-when-cross-origin`
- `Content-Security-Policy` (basic CSP)

**Note**: All security features (CSRF protection, rate limiting, security headers) are already implemented and active. See [SECURITY.md](SECURITY.md) for complete security documentation.

---

## Production Deployment

### 1. Pre-Deployment Checklist

- [ ] `.env` file configured with production values
- [ ] `APP_ENV` set to `"production"`
- [ ] `APP_DEBUG` set to `"false"`
- [ ] Database credentials configured
- [ ] Telegram bot token configured
- [ ] Webhook secret generated and configured
- [ ] HTTPS redirect enabled
- [ ] File permissions set correctly
- [ ] Database migrations run
- [ ] Admin user created
- [ ] Default passwords changed

### 2. Server Requirements

- PHP 7.4 or higher
- MySQL 5.7 or higher (or MariaDB 10.2+)
- Apache with mod_rewrite enabled
- SSL certificate for HTTPS

### 3. PHP Configuration

Recommended `php.ini` settings:

```ini
display_errors = Off
log_errors = On
error_log = /path/to/storage/logs/system.log
session.cookie_httponly = 1
session.cookie_secure = 1
session.use_strict_mode = 1
```

### 4. Apache Configuration

Ensure mod_rewrite is enabled:

```bash
a2enmod rewrite
systemctl restart apache2
```

### 5. Test Deployment

1. **Test Login**
   - Access login page
   - Verify HTTPS redirect works
   - Test login with admin credentials

2. **Test Webhook**
   - Send test payment message
   - Verify transaction appears in system
   - Check logs for errors

3. **Test Security**
   - Try accessing `/app/` directory (should be blocked)
   - Try accessing `.env` file (should be blocked)
   - Test CSRF protection (form submission without token should fail)

---

## Troubleshooting

### Database Connection Issues

**Error**: "Database connection failed"

**Solutions**:
1. Verify database credentials in `.env`
2. Check database server is running
3. Verify user has proper permissions
4. Check firewall rules
5. Test connection manually:
   ```bash
   mysql -u your_user -p your_database
   ```

### Webhook Not Working

**Error**: Webhook requests failing

**Solutions**:
1. Verify `TELEGRAM_WEBHOOK_SECRET` matches Telegram configuration
2. Check webhook URL is accessible via HTTPS
3. Review `/storage/logs/system.log` for errors
4. Verify bot token is correct
5. Test webhook endpoint:
   ```bash
   curl -X POST https://yourdomain.com/public/webhook.php \
     -H "X-Telegram-Bot-Api-Secret-Token: your_secret" \
     -d '{"message":{"text":"test","chat":{"id":123}}}'
   ```

### Login Issues

**Error**: Cannot log in

**Solutions**:
1. Verify user exists in database
2. Check password hash is correct
3. Review rate limiting (may be locked out)
4. Check session directory is writable
5. Review logs for authentication errors

### Permission Errors

**Error**: "Permission denied" or files not writable

**Solutions**:
1. Check file permissions:
   ```bash
   ls -la storage/logs/
   ls -la storage/cache/
   ```
2. Set correct ownership:
   ```bash
   chown -R www-data:www-data storage/
   ```
3. Verify web server user has write access

### HTTPS Redirect Not Working

**Error**: Still accessing via HTTP

**Solutions**:
1. Verify mod_rewrite is enabled
2. Check `.htaccess` file exists and is readable
3. Verify Apache allows `.htaccess` overrides:
   ```apache
   <Directory /path/to/payments>
       AllowOverride All
   </Directory>
   ```
4. Check for conflicting redirects in Apache config

### Error Display in Production

**Error**: Seeing PHP errors on page

**Solutions**:
1. Verify `APP_ENV="production"` in `.env`
2. Verify `APP_DEBUG="false"` in `.env`
3. Check `php.ini`:
   ```ini
   display_errors = Off
   ```
4. Clear any cached configuration

---

## Maintenance

### Regular Tasks

1. **Monitor Logs**
   - Review `/storage/logs/system.log` regularly
   - Check for security issues
   - Monitor failed login attempts

2. **Backup Database**
   ```bash
   mysqldump -u user -p database_name > backup_$(date +%Y%m%d).sql
   ```

3. **Update Dependencies**
   - Keep PHP updated
   - Update server software
   - Review security advisories

4. **Review Access**
   - Audit user accounts
   - Remove unused accounts
   - Review Telegram access list

5. **Check Disk Space**
   - Monitor log file sizes
   - Clean old log entries if needed
   - Monitor cache directory

---

## Support

For additional help:
- Review logs: `/storage/logs/system.log`
- Check security documentation: `SECURITY.md`
- Review code comments in `/app/Helpers/Security.php`

---

**Last Updated**: 2024
**Version**: 1.0

